<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform" 
				version="1.0" 
				xmlns:l="http://www.landxml.org/schema/LandXML-1.2" 
				xmlns:h="http://xml.hexagon.com/schema/HeXML-2.0"
				xmlns:js="urn:mathematics"
                xmlns:msxsl="urn:schemas-microsoft-com:xslt"
                exclude-result-prefixes="l h js">

<!-- ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	-->
<!-- Notice:                                                                                         	-->
<!-- This stylesheet is provided free of charge and all warranties whether implied or by statute or  	-->
<!-- otherwise (including but not limited to fitness for purpose) are specifically excluded to the   	-->
<!-- fullest extent permissible by law. CR Kennedy & Co excludes all liability for any loss howsoever	-->
<!-- incurred to the fullest extent permissible by law. Every effort has been made to assure the     	--> 
<!-- quality of this stylesheet. CR Kennedy & Co provides this Stylesheet "as is" and with all faults	-->
<!-- Permission is hereby granted to use, copy, modify or distribute this stylesheet provided that   	-->
<!-- this notice appears in all copies of the stylesheet and supporting documentation.               	-->
<!-- ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	-->

<!-- ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	-->
<!-- 	Compatibility Requirements																		-->
<!-- 	==========================																		-->
<!-- 		Captivate V3.0 or higher																	-->
<!-- 		Requires SmartWorx Viva V7.0 or higher														-->
<!-- ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	-->

	<xsl:variable name="StylesheetName" select="'12D_V7_0dNZxsl'"/>
	<xsl:variable name="StylesheetUpdate" select="'24/02/23'"/>
	
<!-- ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	-->

<!-- Series Type for String use. 'CAPTIVATE' or 'VIVA'													-->

	<xsl:variable name="Coding" select="'CAPTIVATE'"/>

<!-- Linework Options as set in Captivate and Viva														-->

	<!-- Begin Line          -->	<xsl:variable name="CLbl" select="'B'"/>
	<!-- Begin 3 pt arc      -->	<xsl:variable name="CLbc" select="'BC'"/>
	<!-- Continue line       -->	<xsl:variable name="CLco" select="'C'"/>
	<!-- Begin spline        -->	<xsl:variable name="CLbs" select="'BC'"/>
	<!-- End spline          -->	<xsl:variable name="CLes" select="'EC'"/>
	<!-- Cont spline         -->	<xsl:variable name="CLcs" select="'OC'"/>
	<!-- Close line          -->	<xsl:variable name="CLcl" select="'CLS'"/>

<!-- ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
	
<!-- Name of Stringing Attribute for old Viva coding String Attribute Name								-->

	<xsl:variable name="AttString" select="'String'"/>

<!-- ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->

<!-- Export TPS data as HaVaSd or Points.  Choose one of the following options. 						-->
	
	<!-- select="'HVS'"/>	Observations exported as Code7 Ha/Va/Sd     							    -->
	<!-- select="'ENH'"/>	Observations exported as Code2 E/N/Ht. Use this to export in MGA	    	-->
	
	<xsl:variable name="ExportData" select="'HVS'"/>

<!-- ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->

<!-- 12D Code & Attribute Information.  Choose 'y' or 'n' for the following options.					-->
	
	<!-- Export Code Attributes as Vertex Text       -->	<xsl:variable name="ExportVTxt" select="'n'"/>
	<!-- Export Code Attributes as Attribute Text    -->	<xsl:variable name="ExportATxt" select="'y'"/>
	<!-- Export Obs Meta Data as Attribute Text      -->	<xsl:variable name="ExportMData" select="'y'"/>
	<!-- Export PU with GNSS Obs Meta Data           -->	<xsl:variable name="PUexport" select="'n'"/>
	<!-- Export GNSS Ref Data as Attribute Text      -->	<xsl:variable name="ExportRef" select="'y'"/>
	<!-- Export Hidden Pnt Data as Attribute Text    -->	<xsl:variable name="ExportHid" select="'n'"/>
	<!-- Recalculate Resections in 12D               -->	<xsl:variable name="RxnRecalc" select="'n'"/>
	<!-- Export 12D Op Code 9 - Scale Factor         -->	<xsl:variable name="OpCode9" select="'n'"/>
	<!-- Export Date & Time with point export        -->	<xsl:variable name="ExportDT" select="'n'"/>
	<!-- Export FreeCode Code "c"/Description "d"    -->	<xsl:variable name="ExportFC" select="'c'"/>
	<!-- Export Field note as Vertex Text            -->	<xsl:variable name="ExportFN" select="'n'"/>
	<!-- Export Field note as Vertex Text Name       -->	<xsl:variable name="ExportFNN" select="'Note'"/>
	<!-- Export Fixed Codes for Control              -->	<xsl:variable name="FxdCodeCtrl" select="'n'"/>
	<!-- Export Fixed Codes for Control Code         -->	<xsl:variable name="FxdCodeCtrlC" select="'CTRL'"/>
	<!-- Export Fixed Codes for Station              -->	<xsl:variable name="FxdCodeStn" select="'n'"/>
	<!-- Export Fixed Codes for Station Code         -->	<xsl:variable name="FxdCodeStnC" select="'STN'"/>
	<!-- Export Fixed Codes for Backsight            -->	<xsl:variable name="FxdCodeBs" select="'n'"/>
	<!-- Export Fixed Codes for Backsight Code       -->	<xsl:variable name="FxdCodeBsC" select="'BS'"/>	

<!-- ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~    -->

<!-- GNSS Posit Uncert 95% CL CORS values																-->
	
	<!-- CORS Standard Position PU 95%CL     -->	<xsl:variable name="BasePcl" select="0.008"/>
	<!-- CORS Standard Height   PU 95%CL     -->	<xsl:variable name="BaseHcl" select="0.020"/>
	
	<!-- Formulas Used -->
		<!--Position: Point PU 95%CL = Sqrt( (2dCQ*2.45)Squared + (CORS PU)Squared ) -->
		<!--Height  : Point PU 95%CL = Sqrt( (1dCQ*1.96)Squared + (CORS PU)Squared ) -->

<!-- ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~    -->
<!-- ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	-->

<!-- Output ASCII -->

	<xsl:output method="text" indent="no" encoding="ISO-8859-1"/>
	
<!-- Constants variables read by Onboard exporter-->

	<xsl:variable name="fileExt" select="'fld'"/>
	<xsl:variable name="fileDesc" select="'12D Field File HeXML V2.0'"/>
	
<!-- Configure the default values for distances and angles values -->

	<xsl:variable name="dec1" select="string('0.0')"/>
	<xsl:variable name="dec2" select="string('0.0')"/>
	<xsl:variable name="dec3" select="string('0.000')"/>
	<xsl:variable name="dec4" select="string('0.0000')"/>
	<xsl:variable name="dec5" select="string('0.00000')"/>
	<xsl:variable name="dec6" select="string('0.000000')"/>
	<xsl:variable name="dec8" select="string('0.00000000')"/> 
	<xsl:variable name="StrNo" select="string('0')"/>
	<xsl:variable name="mm" select="string('0.0mm')"/>
	
<!-- Default value when number does not exist -->

	<xsl:decimal-format NaN=""/>	
	
<!-- Columns separators for the ASCII file -->

	<xsl:variable name="spaceSep" select="' '"/>
	<xsl:variable name="commaSep" select="','"/>
	<xsl:variable name="semicolSep" select="';'"/>
	<xsl:variable name="colSep" select="':'"/>
	<xsl:variable name="tabSep" select="'&#x9;'"/>
	<xsl:variable name="returnSep" select="'&#13;&#10;'"/>
	<xsl:variable name="divline" select="concat('//------------------------------------------------------------------------------------------',$returnSep)"/>
	
<!-- Default Variable for Target Height -->

	<xsl:variable name="defaultHT" select="''"/>
	
<!-- Keys -->

	<xsl:key name="KeyPointH" match="h:Point" use="@uniqueID"/>
	<xsl:key name="KeyRawObsL" match="l:RawObservation" use="l:TargetPoint/@pntRef"/>
	<xsl:key name="KeyRawObsH" match="h:RawObservation" use="@targetPntRef"/>
	<xsl:key name="KeyGPSPosL" match="l:GPSPosition" use="@pntRef"/>
	<xsl:key name="KeyGPSPosH" match="h:GPSPosition" use="@targetPntRef"/>
	<xsl:key name="KeyTPSCorH" match="h:TPSCorrection" use="@uniqueID"/>
	<xsl:key name="KeyCgPointL" match="l:CgPoint" use="@name"/>
	<xsl:key name="KeyInstSetupH" match="h:InstrumentSetup" use="@uniqueID"/>
	<xsl:key name="KeyInstSetupL" match="l:InstrumentSetup" use="@id"/>
	<xsl:key name="KeyLinesH" match="h:CoordGeom/*" use="@oID"/>
	<xsl:key name="KeyApplH" match="h:HexagonLandXML/h:ApplicationResults/*" use="@ApplicationNumber"/>
	<xsl:key name="KeyCodeH" match="h:Code" use="@codeName"/>
	<xsl:key name="KeyAttributeH" match="h:Code/h:PointCode" use="@codeName"/>
	
	
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
<!--	Main Template																					-->
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
	<xsl:template match="/l:LandXML">
		
		<!-- Process Linework for Captivate -->
		<xsl:if test="$Coding = 'CAPTIVATE'">
			<xsl:call-template name="LineworkExtract"/>
		</xsl:if>
		
		<!-- Process File Header -->
		<xsl:call-template name="FileHeader"/>
		
				<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  -->
                  <!-- |	NZ Application Survey	- GNSS BASE EXPORT									| -->
                  <!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  -->
		
		<xsl:for-each select="//h:Point[@subclass = 'positionAndHeight' and @class = 'reference']"> 
			<xsl:variable name="GPSPointH" select="key('KeyPointH', @uniqueID)"/>
		    <xsl:variable name="GPSCgPoint" select="key('KeyCgPointL', @uniqueID)"/>
		    <xsl:variable name="GPSPositionL" select="key('KeyGPSPosL', @uniqueID)"/>
		    <xsl:variable name="GPSPos" select="key('KeyGPSPosH', @uniqueID)"/>
			
		<!-- CR/LF       -->	<xsl:value-of select="$returnSep"/>
		<!-- Line Beg    -->	<xsl:value-of select="concat(' 2', $tabSep, $tabSep)"/>
		<!-- code     	 -->	<xsl:value-of select="concat('Base', $tabSep, $tabSep)"/>
		<!-- Point ID    -->	<xsl:value-of select="concat($GPSCgPoint/@oID, $tabSep, $tabSep)"/>
			                    <xsl:value-of select="concat($tabSep, format-number($GPSPointH/h:Coordinates/h:Local/h:Grid/@e, $dec3))"/>
		<!-- Northing    -->	<xsl:value-of select="concat($tabSep, format-number($GPSPointH/h:Coordinates/h:Local/h:Grid/@n, $dec3))"/>
		<!-- Height      -->	<xsl:choose>
									<xsl:when test="$GPSPointH/h:Coordinates/h:Local/h:Grid/@hghthO">
										<xsl:value-of select="concat($tabSep, format-number($GPSPointH/h:Coordinates/h:Local/h:Grid/@hghthO, $dec3))"/>
									</xsl:when>
									<xsl:otherwise>
										<xsl:value-of select="concat($tabSep, format-number($GPSPointH/h:Coordinates/h:Local/h:Grid/@hghtE, $dec3))"/>
									</xsl:otherwise>
								</xsl:choose>
		<!-- CR/LF       -->	<xsl:value-of select="$returnSep"/>		
							<xsl:if test="$ExportRef ='y'">
						<!-- Start-Ref       -->	<xsl:value-of select="concat('124', $tabSep, $tabSep, $tabSep, 'GNSS Reference Information', $tabSep, '0', $returnSep)"/>


						<!-- Easting         -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Easting', $tabSep, format-number($GPSPointH/h:Coordinates/h:Local/h:Grid/@e,$dec3), $returnSep)"/>
						<!-- Northing        -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Northing', $tabSep, format-number($GPSPointH/h:Coordinates/h:Local/h:Grid/@n,$dec3), $returnSep)"/>
						<!-- Orth Height     -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Orthometric Height', $tabSep, format-number($GPSPointH/h:Coordinates/h:Local/h:Grid/@hghthO,$dec3), $returnSep)"/>
						<!-- Latitude        -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Latitude', $tabSep, js:DDtoDMSTxt(string($GPSPointH/h:Coordinates/h:Local/h:Geodetic/@lat),5), $returnSep)"/>
						<!-- Longitude       -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Longitude', $tabSep, js:DDtoDMSTxt(string($GPSPointH/h:Coordinates/h:Local/h:Geodetic/@lon),5), $returnSep)"/>
						<!-- Ellipsoidal Ht  -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Ellipsoidal Height', $tabSep, format-number($GPSPointH/h:Coordinates/h:Local/h:Geodetic/@hghtE,$dec3), $returnSep)"/>


						<!-- End-GPS         -->	<xsl:value-of select="concat('125', $tabSep, $tabSep, $tabSep, $tabSep, '0', $returnSep)"/>
					</xsl:if>
		
		
		</xsl:for-each>
		
	              <!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  -->
                  <!-- |	NZ Application Survey	- Gs18i Image output							| -->
                  <!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~  -->
		<xsl:value-of select="concat('//-------Points determined from Imagery------------', $returnSep)"/>
		<xsl:for-each select="//h:Point[@subclass = 'imaging' and @class = 'measured']"> 
			<xsl:variable name="GPSPointH" select="key('KeyPointH', @uniqueID)"/>
		    <xsl:variable name="GPSCgPoint" select="key('KeyCgPointL', @uniqueID)"/>
		    <xsl:variable name="GPSPositionL" select="key('KeyGPSPosL', @uniqueID)"/>
		    <xsl:variable name="GPSPos" select="key('KeyGPSPosH', @uniqueID)"/>
			
		<!-- CR/LF       -->	<xsl:value-of select="$returnSep"/>
		<!-- Line Beg    -->	<xsl:value-of select="concat(' 2', $tabSep, $tabSep)"/>
		<!-- code     	 -->	<xsl:value-of select="concat($GPSPointH/h:PointCode/@code, $tabSep, $tabSep)"/>
		<!-- Point ID    -->	<xsl:value-of select="concat($GPSCgPoint/@oID, $tabSep, $tabSep)"/>
			                    <xsl:value-of select="concat($tabSep, format-number($GPSPointH/h:Coordinates/h:Local/h:Grid/@e, $dec3))"/>
		<!-- Northing    -->	<xsl:value-of select="concat($tabSep, format-number($GPSPointH/h:Coordinates/h:Local/h:Grid/@n, $dec3))"/>
		<!-- Height      -->	<xsl:choose>
									<xsl:when test="$GPSPointH/h:Coordinates/h:Local/h:Grid/@hghthO">
										<xsl:value-of select="concat($tabSep, format-number($GPSPointH/h:Coordinates/h:Local/h:Grid/@hghthO, $dec3))"/>
									</xsl:when>
									<xsl:otherwise>
										<xsl:value-of select="concat($tabSep, format-number($GPSPointH/h:Coordinates/h:Local/h:Grid/@hghtE, $dec3))"/>
									</xsl:otherwise>
								</xsl:choose>
		<!-- CR/LF       -->	<xsl:value-of select="$returnSep"/>	
				<!-- Surv Att        -->	<xsl:value-of select="concat('124', $tabSep, $tabSep, $tabSep, 'Survey Attributes', $tabSep, '0', $returnSep)"/>
												<xsl:value-of select="concat('73', $tabSep, $tabSep, 'SurveyClass', $tabSep, 'Image Capture', $returnSep)"/>
												<xsl:value-of select="concat('73', $tabSep, $tabSep, '3d CQ', $tabSep, format-number($GPSPointH/h:PointQuality/@CQ3D, $dec3), $returnSep)"/>
				<!-- End-Surv Att    -->	<xsl:value-of select="concat('125', $tabSep, $tabSep, $tabSep, $tabSep, '0', $returnSep)"/>
		</xsl:for-each>	
			
		
		<xsl:value-of select="concat('//------------------------------------------------------------------------------------------', $returnSep)"/>	
		
		<!-- Process observations -->
		<xsl:apply-templates select="l:CgPoints/l:CgPoint [@role = 'measured'] | 
									l:Survey/l:InstrumentSetup/l:InstrumentPoint | 
									h:HexagonLandXML/h:FreeCodes/h:FreeCode | 
									h:HexagonLandXML/h:ApplicationResults/h:ApplicationCheckRecordedPoint | 
									h:HexagonLandXML/h:ApplicationResults/h:ApplicationGPSHiddenPoint | 
									h:HexagonLandXML/h:ApplicationResults/h:ApplicationMeasureForesight">
		  <xsl:sort select="@timeStamp | @ApplicationStartDateTime " order="ascending"/>
		</xsl:apply-templates>
	
	</xsl:template>
	
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
<!--	File Header Export																				-->
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
	<xsl:template name="FileHeader">
		<!-- File Header Information -->
		<xsl:value-of select="concat('{Version 6.0}', $returnSep)"/>
		<xsl:value-of select="$divline"/>
		<xsl:value-of select="concat('// ', $returnSep)"/>
		<xsl:value-of select="concat('// ', $fileDesc, $returnSep)"/>
		<xsl:value-of select="concat('// ', 'Style Sheet  : ', $StylesheetName, $returnSep)"/>
		<xsl:value-of select="concat('// ', 'Revision Date: ', $StylesheetUpdate, $returnSep)"/>
		<xsl:value-of select="concat('// ', 'Code List:' , //h:Codelist/@codelistName,$returnSep)"/>
		<xsl:value-of select="concat('// ', $returnSep)"/>
		
		<xsl:if test="//l:LandXML/l:Survey/l:Equipment/l:InstrumentDetails">
			<xsl:value-of select="concat('// ', 'Instrument   : ', //l:LandXML/l:Survey/l:Equipment/l:InstrumentDetails/@model, '  #',//l:LandXML/l:Survey/l:Equipment/l:InstrumentDetails/@serialNumber, $returnSep)"/>
		</xsl:if>
		<xsl:if test="//h:Survey/h:TPSCalibration">
			<xsl:value-of select="concat('// ', 'Last Chk Adj : ', js:XMLTimeAuDate(string(//h:Survey/h:TPSCalibration/@timeStamp)),' ', js:XMLTimeAuTime(string(//h:Survey/h:TPSCalibration/@timeStamp)), $returnSep)"/>
		</xsl:if>
		<xsl:if test="//l:LandXML/l:Survey/l:Equipment/l:GPSReceiverDetails">
			<xsl:value-of select="concat('// ', 'GNSS Model   : ', //l:LandXML/l:Survey/l:Equipment/l:GPSReceiverDetails/@model, '  #',//l:LandXML/l:Survey/l:Equipment/l:GPSReceiverDetails/@serialNumber, $returnSep)"/>
		</xsl:if>
		<xsl:value-of select="concat('// ', 'Job name     : ', substring-before(//h:Survey/@name,'_'), $returnSep)"/>
		<xsl:value-of select="concat('// ', 'Description 1: ', //h:Survey/@Description1, $returnSep)"/>
		<xsl:value-of select="concat('// ', 'Description 2: ', //h:Survey/@Description2, $returnSep)"/>
		<xsl:value-of select="concat('// ', 'Surveyor     : ', //h:Survey/@Creator, $returnSep)"/>
		<xsl:value-of select="concat('// ', $returnSep)"/>
		
		<xsl:value-of select="concat('// ', 'Coordinate System: ', //l:CoordinateSystem/@name, $returnSep)"/>
		<xsl:if test="//l:CoordinateSystem/@name !='None'">
			<xsl:value-of select="concat('// ', 'Projection       : ', //l:CoordinateSystem/@projectedCoordinateSystemName, $returnSep)"/>
			<xsl:value-of select="concat('// ', 'Ellipsoid        : ', //l:CoordinateSystem/@ellipsoidName, $returnSep)"/>
		</xsl:if>
		<xsl:value-of select="concat('// ', $returnSep)"/>
		
		<xsl:value-of select="$divline"/>
	</xsl:template>

<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
<!--	CgPoints Export																					-->
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
	<xsl:template match="l:CgPoint">
		<xsl:variable name="CgPoint" select="."/>
		<xsl:variable name="RawObsL" select ="key('KeyRawObsL', $CgPoint/@name)"/>
		<xsl:variable name="PointH" select ="key('KeyPointH', $CgPoint/@name)"/>
		<xsl:variable name="CAppSvy" select="key('KeyApplH',$PointH/@applicationNumber)"/>		
		
		<xsl:choose>
			<xsl:when test="($PointH/@subclass = 'TPS')">
					<xsl:if test="($RawObsL/@purpose = 'normal') and ($CAppSvy/@ApplicationName != 'ApplMeasureForesight')">
						<xsl:call-template name="ObsExport">
							<xsl:with-param name="CgPointName" select="$CgPoint/@name"/>
							<xsl:with-param name="SurveyObs" select="'y'"/>
							<xsl:with-param name="ObsType" select="' 7'"/>
						</xsl:call-template>
					</xsl:if>
				</xsl:when>
			<xsl:otherwise>
				<xsl:call-template name="ObsExport">
					<xsl:with-param name="CgPointName" select="$CgPoint/@name"/>
					<xsl:with-param name="SurveyObs" select="'y'"/>
				</xsl:call-template>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
<!--	Observation Export																				-->
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
	<xsl:template name="ObsExport">
		<xsl:param name="CgPointName"/>
		<xsl:param name="SurveyObs"/>
		<xsl:param name="ObsType"/>
		<xsl:variable name="CgPoint" select ="key('KeyCgPointL', $CgPointName)"/>
		<xsl:variable name="PointH" select ="key('KeyPointH', $CgPointName)"/>
		<xsl:variable name="RawObsL" select ="key('KeyRawObsL', $CgPointName)"/>
		<xsl:variable name="RawObsH" select ="key('KeyRawObsH', $CgPointName)"/>
		<xsl:variable name="GPSPosL" select ="key('KeyGPSPosL', $CgPointName)"/>
		<xsl:variable name="GPSPosH" select ="key('KeyGPSPosH', $CgPointName)"/>
		<xsl:variable name="TPSCorH" select="key('KeyTPSCorH',$RawObsH/@tpsCorrectionRef)"/>
	
		<!-- TPS Stn & GPS Ref   -->
		<xsl:choose>
		
			<!-- TPS Specific Checks -->
			<xsl:when test="$PointH/@subclass = 'TPS'">
				
				<!-- Scale/AtmosPPM  -->
				<xsl:variable name="TPScorrStatus" select="js:TPSCorr(number($TPSCorH/@uniqueID))"/>
				<xsl:if test="$TPScorrStatus = 1">
					<xsl:value-of select="concat('//',$tabSep,$tabSep,'Atmospheric PPM: ',format-number($TPSCorH/@atmosphericPPM,$dec1),$tabSep,'Temp: ',format-number($TPSCorH/@dryTemperature,$dec1),$tabSep,'Press: ',format-number($TPSCorH/@pressure,$dec1),$tabSep,'Geometric PPM: ',format-number($TPSCorH/@individualPPM,$dec1),$tabSep,'(sf): ',format-number(($TPSCorH/@individualPPM + 1000000) div 1000000,$dec6),$returnSep)"/>
				</xsl:if>
				
				<!-- Current TPS     -->
				<xsl:variable name="edmatr_TPS_Meas" select="js:checkEDMATR(string($RawObsH/@edmKind), string($RawObsH/@edmMode), string($RawObsH/@atrMode), string($RawObsL/@reflectorName), string($RawObsH/@reflectorConstant))"/>
				<xsl:value-of select="$edmatr_TPS_Meas"/>
				
				<!-- Target Ht Check & write only on change -->
				<xsl:variable name="currentTargetHt" select="js:CurrentTHCheck(number($RawObsL/@targetHeight))"/>
				<xsl:value-of select="string($currentTargetHt)"/>
				
				<!-- Tgt Ht Line on every observation -->
				<!--<xsl:value-of select="concat(' 5', $tabSep, $tabSep,format-number($RawObsL/@targetHeight, $dec3), $returnSep)"/>-->
				
			</xsl:when>
			
			<!-- GPS Specific Checks -->
			<xsl:otherwise>
				<!-- Ant Ht Check    -->	<xsl:variable name="currentAntHt" select="js:CurrentAntCheck(number($GPSPosL/../@antennaHeight))"/>
				<!-- Ant Ht Line     -->	<xsl:value-of select="string($currentAntHt)"/>
			</xsl:otherwise>
			
		</xsl:choose>
	
		<!-- Meas Line Start     -->
		<xsl:choose>
			<xsl:when test="($PointH/@subclass = 'TPS') and ($ExportData = 'HVS')">
				<xsl:value-of select="concat($ObsType, $tabSep)"/>
			</xsl:when>
			<xsl:otherwise>
				<xsl:value-of select="concat(' 2', $tabSep)"/>
			</xsl:otherwise>
		</xsl:choose>
	
		<!-- Export Time Option  -->
			<xsl:if test="$ExportDT = 'y'">
				<xsl:value-of select="concat(js:XMLTimeAuDate(string($CgPoint/@timeStamp)),$spaceSep,js:XMLTimeAuTime(string($CgPoint/@timeStamp)))"/>
			</xsl:if>
			<xsl:value-of select="$tabSep"/>
	
		<!-- code & String		 -->
			<xsl:choose>
				<xsl:when test="($FxdCodeBs = 'y') and ($SurveyObs = 'n')">
					<xsl:value-of select="$FxdCodeBsC"/>
					<xsl:value-of select="concat($tabSep, $tabSep)"/>
				</xsl:when>
				<xsl:otherwise>
					<xsl:value-of select="$PointH/h:PointCode/@code"/>
					<xsl:value-of select="$tabSep"/>
					<xsl:call-template name="GetString">
						<xsl:with-param name="CgPointName" select="$CgPointName"/>
					</xsl:call-template>
					<xsl:value-of select="$tabSep"/>
				</xsl:otherwise>
			</xsl:choose>

		<!-- Point ID			 -->
			<xsl:value-of select="concat($CgPoint/@oID, $tabSep, $tabSep)"/>
	
		<!-- VTXT Att Export     -->
			<xsl:if test="$ExportVTxt = 'y'">
				<xsl:for-each select="$PointH/h:PointCode/h:Attribute">
				<xsl:variable name="attPointCode" select="."/>
					<xsl:if test="$attPointCode/@name != $AttString">
						<xsl:value-of select="concat($attPointCode/@value, ' ')"/>
					</xsl:if>	
				</xsl:for-each>
			</xsl:if>
			
		<!-- VTXT Field Note Export     -->
			<xsl:if test=" ($ExportVTxt = 'n') and ($ExportFN = 'y')">
				<xsl:for-each select="$PointH/h:PointCode/h:Attribute">
				<xsl:variable name="attPointCode" select="."/>
					<xsl:if test="$attPointCode/@name = $ExportFNN">
						<xsl:value-of select="concat($attPointCode/@value, ' ')"/>
					</xsl:if>	
				</xsl:for-each>
			</xsl:if>
	
		<!-- Observation Data    -->
			<xsl:choose>
				<xsl:when test="($PointH/@subclass = 'TPS') and ($ExportData = 'HVS')">
					<!-- Ha  	<xsl:value-of select="concat( $tabSep, format-number(js:ConvertDMStoDD(number($RawObsL/@horizAngle)), $dec6), $tabSep)"/> -->
		<xsl:variable name="inSetRnd" select="format-number($RawObsL/@horizAngle,$dec4 )" />
		<xsl:variable name="BRD" select="substring-before($inSetRnd, '.')"/>
		<xsl:variable name="BRM" select="substring($inSetRnd, string-length($inSetRnd) - 3,2)"/>
		<xsl:variable name="BRMdd" select="format-number(($BRM div 60),$dec8)"/>	
		<xsl:variable name="BRS" select="substring($inSetRnd, string-length($inSetRnd) - 1,2)"/>
		<xsl:variable name="BRSdd" select="format-number(($BRS div 3600),$dec8)"/>
		<xsl:variable name="BRDdd" select="format-number(($BRD + $BRMdd + $BRSdd),$dec6)"/>
		<xsl:value-of select= "concat($tabSep,$BRDdd, $tabSep)" />
		
					<!-- Va  	<xsl:value-of select="concat(format-number(js:ConvertDMStoDD(number($RawObsL/@zenithAngle)), $dec6), $tabSep)"/>	-->
		<xsl:variable name="inSetVertRnd" select="format-number($RawObsL/@zenithAngle,$dec4 )" />
		<xsl:variable name="VertBRD" select="substring-before($inSetVertRnd, '.')"/>
		<xsl:variable name="VertBRM" select="substring($inSetVertRnd, string-length($inSetVertRnd) - 3,2)"/>
		<xsl:variable name="VertBRMdd" select="format-number(($VertBRM div 60),$dec8)"/>	
		<xsl:variable name="VertBRS" select="substring($inSetVertRnd, string-length($inSetVertRnd) - 1,2)"/>
		<xsl:variable name="VertBRSdd" select="format-number(($VertBRS div 3600),$dec8)"/>
		<xsl:variable name="VertBRDdd" select="format-number(($VertBRD + $VertBRMdd + $VertBRSdd),$dec6)"/>
		<xsl:value-of select= "concat($VertBRDdd, $tabSep)" /> 
		
					<!-- Sd  -->	<xsl:if test="$RawObsL/@slopeDistance">
										<xsl:value-of select="format-number($RawObsL/@slopeDistance + ($TPSCorH/@atmosphericPPM div 1000000 * $RawObsL/@slopeDistance), $dec3)"/>
									</xsl:if>
				</xsl:when>
				<xsl:otherwise>
					<!-- E   -->	<xsl:value-of select="concat($tabSep, format-number($PointH/h:Coordinates/h:Local/h:Grid/@e, $dec3))"/>
					<!-- N   -->	<xsl:value-of select="concat($tabSep, format-number($PointH/h:Coordinates/h:Local/h:Grid/@n, $dec3))"/>
					<!-- H   -->	<xsl:choose>
										<xsl:when test="$PointH/h:Coordinates/h:Local/h:Grid/@hghthO">
											<xsl:value-of select="concat($tabSep, format-number($PointH/h:Coordinates/h:Local/h:Grid/@hghthO, $dec3))"/>
										</xsl:when>
										<xsl:otherwise>
											<xsl:value-of select="concat($tabSep, format-number($PointH/h:Coordinates/h:Local/h:Grid/@hghtE, $dec3))"/>
										</xsl:otherwise>
									</xsl:choose>
				</xsl:otherwise>
			</xsl:choose>
		
		<!-- CR/LF               -->
			<xsl:value-of select="$returnSep"/>
		
		<xsl:if test="$SurveyObs = 'y'">
		
			<!-- Code Attributes     -->
				<xsl:if test="$ExportATxt = 'y'">
					
					<xsl:call-template name="CodeAtt">
						<xsl:with-param name="PointH" select="$PointH"/>
					</xsl:call-template>
				</xsl:if>
			
			<!-- Obs Attributes      -->
				<xsl:if test="$ExportMData = 'y'">
					<xsl:call-template name="ObsAtt">
						<xsl:with-param name="PointH" select="$PointH"/>
						<xsl:with-param name="CgPointName" select="$CgPointName"/>
					</xsl:call-template>
				</xsl:if>
				
			<!-- TPS Offsets Chk     -->
				<xsl:call-template name="TPSoffsets">
					<xsl:with-param name="CgPointName" select="$CgPointName"/>
				</xsl:call-template>	
			
			<!-- Close String Chk    -->
				<xsl:call-template name="GetCloseString">
					<xsl:with-param name="CgPointName" select="$CgPointName"/>
				</xsl:call-template>
			
		</xsl:if>
		
		<xsl:value-of select="js:GetTinnableFlag()"/>
		<xsl:value-of select="js:TinnableFlag(number(1))"/>
		
	</xsl:template>

<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
<!--	Measure Foresight																				-->
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
	<xsl:template match="h:ApplicationMeasureForesight">
		<xsl:variable name="ApplicationMeasureForesight" select="."/>
			
		<xsl:choose>
			<xsl:when test="@measFsPointRole = 'Backsight'">
							<xsl:value-of select="$divline"/>
				<xsl:value-of select="concat('//',$tabSep,$tabSep,'Measure Forward Station - Backsight Observation',$returnSep)"/>
				<xsl:value-of select="$divline"/>
				<xsl:call-template name="ObsExport">
					<xsl:with-param name="CgPointName" select="./h:MeasFsRawData/@measFsMeasPointID"/>
					<xsl:with-param name="SurveyObs" select="'n'"/> <!-- 'n' will not export Attribute (Code & TPS) information with the BS Observations -->
					<xsl:with-param name="ObsType" select="'//'"/>
				</xsl:call-template>
			</xsl:when>
			<xsl:otherwise>
				<xsl:value-of select="$divline"/>
				<xsl:value-of select="concat('//',$tabSep,$tabSep,'Measure Forward Station - Foresight Observation',$returnSep)"/>
				<xsl:value-of select="$divline"/>
				<xsl:call-template name="ObsExport">
					<xsl:with-param name="CgPointName" select="./h:MeasFsRawData/@measFsMeasPointID"/>
					<xsl:with-param name="SurveyObs" select="'y'"/> <!-- 'n' will not export Attribute (Code & TPS) information with the BS Observations -->
					<xsl:with-param name="ObsType" select="' 7'"/>
				</xsl:call-template>
			</xsl:otherwise>
		</xsl:choose>
		
		<xsl:value-of select="concat('//',$returnSep,'//',$tabSep,$tabSep,'Residuals Information',$returnSep)"/>
		<xsl:value-of select="concat('//',$tabSep,$tabSep,$tabSep,$tabSep,'Std.D Ha(dms): ',format-number(@measFsHzArcStdDev,$dec5), $tabSep, 'Std.D Va(dms): ',format-number(@measFsVStdDev,$dec5),$tabSep, 'Std.D Sd(m): ',format-number(@measFsDistStdDev,$dec4), $returnSep)"/>
		
		<xsl:if test="@measFsNumberSets !=1">
				<xsl:for-each select="./h:MeasFsRawData">
					<xsl:variable name="RawObsH" select ="key('KeyRawObsH', @measFsMeasPointID)"/>
					<xsl:value-of select="concat('//',$tabSep,$tabSep,$tabSep,$tabSep,'Set: ',js:jText(string($RawObsH/h:MeasureForesight/@setNumber),2,'R','N'),'  ')"/>
					<xsl:choose>
						<xsl:when test="$RawObsH/h:MeasureForesight/@face2PntRef">
							<xsl:value-of select="'Meas: 2 Face Avg'"/>
						</xsl:when>
						<xsl:otherwise>
							<xsl:value-of select="'Meas: 1 Face Obs'"/>
						</xsl:otherwise>
					</xsl:choose>
					<xsl:value-of select="concat('  HA=',js:jAngle(number($RawObsH/h:MeasureForesight/@hzAverage),8,'R',4))"/>
					<xsl:value-of select="concat('  VA=',js:jAngle(number($RawObsH/h:MeasureForesight/@vAverage),8,'R',4))"/>
					<xsl:value-of select="concat('  SD=',js:jDist(number($RawObsH/h:MeasureForesight/@distAverage),7,'R',3))"/>
					<xsl:value-of select="$returnSep"/>
					
				</xsl:for-each>
				<xsl:value-of select="concat('//', $returnSep)"/>
			</xsl:if>
			
		<xsl:value-of select="$divline"/>
		<!-- Reset the TargetHt to default --> <xsl:value-of select="js:ResetCurrentTH(number($defaultHT))"/>
		<!-- Reset the EDMATR to default   --> <xsl:value-of select="js:ResetEDMATR()"/>
				
	</xsl:template>

<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
<!--	GPS Hidden Point																				-->
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
	<xsl:template match="h:ApplicationGPSHiddenPoint">
		<xsl:variable name="ApplicationGPSHiddenPoint" select="."/>
		<xsl:if test="$ExportHid = 'y'">
			<xsl:value-of select="concat('124', $tabSep, $tabSep, $tabSep, 'GNSS Hidden Point', $tabSep, '0', $returnSep)"/>
			<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Date', $tabSep,js:XMLTimeAuDate(string(@ApplicationStartDateTime)), $returnSep)"/>
			<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Time', $tabSep,js:XMLTimeAuTime(string(@ApplicationStartDateTime)), $returnSep)"/>
			<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Method', $tabSep, @GPSHiddenPointMethod, $returnSep)"/>
			<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Base Point', $tabSep, @GPSHiddenPointHeightPoint1, $returnSep)"/>
			<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Base Point Height', $tabSep, format-number(@GPSHiddenPointInstrumentHeight1,$dec3), $returnSep)"/>
			<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Target Point Height', $tabSep, format-number(@GPSHiddenPointTargetHeight1, $dec3), $returnSep)"/>
			<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Horiz Angle', $tabSep, js:DDtoDMSTxt(string(js:ConvertDMStoDD(string(@GPSHiddenPointBearing))),0), $returnSep)"/>
			<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Vert Angle', $tabSep, js:DDtoDMSTxt(string(90 + js:ConvertDMStoDD(string(@GPSHiddenPointElevationAngle))),0), $returnSep)"/>
			<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Slope Dist', $tabSep, format-number(@GPSHiddenPointSlopeDistance, $dec3), $returnSep)"/>
			<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Horiz Dist', $tabSep, format-number(@GPSHiddenPointHzDistance, $dec3), $returnSep)"/>
			<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Vert Dist', $tabSep, format-number(@GPSHiddenPointDiffHeight1, $dec3), $returnSep)"/>
			<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Height Diff', $tabSep, format-number(@GPSHiddenPointInstrumentHeight1 + @GPSHiddenPointDiffHeight1, $dec3), $returnSep)"/>
			<xsl:value-of select="concat('125', $tabSep, $tabSep, $tabSep, $tabSep, '0', $returnSep)"/>
		</xsl:if>
	</xsl:template>

<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
<!--	Check Recorded Point																			-->
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
	<xsl:template match="h:ApplicationCheckRecordedPoint">
		<xsl:variable name="ApplicationCheckRecordedPoint" select="."/>
		
		<xsl:variable name="CheckRecPtPointIDOid" select="key('KeyCgPointL', @CheckRecPtPointID)"/>
		
		<!-- Section Specific Java Scripts -->
		
		<xsl:variable name="currentTargetHt" select="js:CurrentTHCheck(number(@targetHeight))"/>
		
		<!-- XMLTimeAuDate-->	<xsl:variable name="auDate" select="js:XMLTimeAuDate(string(@ApplicationStartDateTime))"/>
		<!-- XMLTimeAuTime-->	<xsl:variable name="auTime" select="js:XMLTimeAuTime(string(@ApplicationStartDateTime))"/>
		<!-- Ha DMS to DD -->	<xsl:variable name="angleDD_TPS_Chk_Ha" select="js:ConvertDMStoDD(number(@CheckRecPtMeasAzimuth))"/>
		<!-- Va DMS to DD -->	<xsl:variable name="angleDD_TPS_Chk_Va" select="js:ConvertDMStoDD(number(@CheckRecPtMeasVAngle))"/>	
    										
		<!-- Line Begin   -->	<xsl:value-of select="concat(	'//', $returnSep)"/>
		<!-- Divide Line  -->	<xsl:value-of select="$divline"/>
		<!-- PTS Line     -->	<xsl:if test="$ExportData = 'PTS'">//</xsl:if>
		<!-- BS Target Ht -->	<xsl:value-of select="concat(' 5', $tabSep, $tabSep, format-number(@targetHeight, $dec3), $returnSep)"/>
		<!-- PTS Line     -->	<xsl:if test="$ExportData = 'PTS'">//</xsl:if>
		<!-- Line Begin   -->	<xsl:value-of select="concat(' 6', $tabSep)"/>
		<!-- Export Time Option  -->
								<xsl:if test="$ExportDT = 'y'">
									<xsl:value-of select="concat(js:XMLTimeAuDate(string(@ApplicationStartDateTime)),$spaceSep,js:XMLTimeAuTime(string(@ApplicationStartDateTime)))"/>
								</xsl:if>
								<xsl:value-of select="$tabSep"/>
		<!-- Line Con't   -->	<xsl:value-of select="concat($tabSep, $tabSep)"/>	
		<!-- Point ID     -->	<xsl:value-of select="concat($CheckRecPtPointIDOid/@oID, $tabSep, $CheckRecPtPointIDOid/@oID, $tabSep, $tabSep)"/>
		<!-- Meas Ha      -->	<xsl:value-of select="concat(format-number(string($angleDD_TPS_Chk_Ha), $dec6), $tabSep)"/>
		<!-- Meas Va      -->	<xsl:value-of select="format-number(string($angleDD_TPS_Chk_Va), $dec6)"/>
		<!-- Meas Sd      -->	<xsl:value-of select="concat( $tabSep, format-number(@CheckRecPtMeasSlopeDist, $dec3))"/>
		<!-- CR/LF        -->	<xsl:value-of select="$returnSep"/>
  		
		<!-- Divide Line  -->	<xsl:value-of select="$divline"/>
		<!-- Info 1       -->	<xsl:value-of select="concat('//', $tabSep, $tabSep, 'Check Shot observed from Station: ')"/>
		<!-- Stn Info     -->	<xsl:value-of select="concat(@CheckRecPtStationID, $tabSep, string($auDate), $tabSep, string($auTime), $returnSep)"/>
		<!-- Ref Object   -->	<xsl:value-of select="concat('//', $tabSep, $tabSep, 'Check Point: ',$CheckRecPtPointIDOid/@oID, $returnSep)"/>
		<!-- Calculated   -->	<xsl:value-of select="concat('//', $tabSep, $tabSep, 'Calculated  :',$tabSep, 'Hz(dms): ', format-number(@CheckRecPtCalcAzimuth, $dec4),  $tabSep, 'Hd: ', format-number(@CheckRecPtCalcHzDist,$dec3),  $tabSep,'dHt: ', format-number(@CheckRecPtCalcHtDiff,$dec3),  $returnSep)"/>
		<!-- Measured     -->	<xsl:value-of select="concat('//', $tabSep, $tabSep, 'Measured    :',$tabSep, 'Hz(dms): ', format-number(@CheckRecPtMeasAzimuth, $dec4),  $tabSep, 'Hd: ', format-number(@CheckRecPtMeasHzDist,$dec3),  $tabSep,'dHt: ', format-number(@CheckRecPtMeasHtDiff,$dec3),  $returnSep)"/>
		<!-- Differences  -->	<xsl:value-of select="concat('//', $tabSep, $tabSep, 'Differences :',$tabSep, 'Hz(dms): ', format-number(@CheckRecPtDeltaAzimuth, $dec4), $tabSep, 'Hd: ', format-number(@CheckRecPtDeltaHzDist,$dec3), $tabSep,'dHt: ', format-number(@CheckRecPtDeltaHtDiff,$dec3), $returnSep)"/>
		<!-- Divide Line  -->	<xsl:value-of select="$divline"/>
  		
		<!-- Reset the TargetHt to default --> <xsl:value-of select="js:ResetCurrentTH(number($defaultHT))"/>
		<!-- Reset the EDMATR to default   --> <xsl:value-of select="js:ResetEDMATR()"/>

	</xsl:template>

<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
<!--	TPS Setup																						-->
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
	<xsl:template match="l:InstrumentPoint">
		<xsl:variable name="InstPointL" select="."/>
		<xsl:variable name="InstSetupL" select="key('KeyInstSetupL',$InstPointL/../@id)"/>
		<xsl:variable name="InstSetupH" select="key('KeyInstSetupH',$InstPointL/../@id)"/>
		<xsl:variable name="TPSCorH" select="key('KeyTPSCorH',$InstSetupH/h:RawObservation[1]/@tpsCorrectionRef)"/>
		<xsl:variable name="PointH" select="key('KeyPointH',$InstSetupL/l:InstrumentPoint/@pntRef)"/>
		<xsl:variable name="auDate" select="js:XMLTimeAuDate(string($InstSetupL/l:InstrumentPoint/@timeStamp))"/>
		<xsl:variable name="auTime" select="js:XMLTimeAuTime(string($InstSetupL/l:InstrumentPoint/@timeStamp))"/>
		
		<xsl:value-of select="$divline"/>
		<xsl:value-of select="concat('// Station Setup - ', $InstSetupH/@tpsSetupMethod, $tabSep, $auDate, ' ', $auTime, $returnSep)"/>
		
		<xsl:if test="$OpCode9 = 'y'">
			<xsl:value-of select="concat(' 9', $tabSep, $tabSep, format-number(($TPSCorH/@individualPPM + 1000000) div 1000000,$dec6),$returnSep)"/>
			</xsl:if>
		
		<xsl:call-template name="CtrlExport">
					<xsl:with-param name="CtrlPnt" select="$InstSetupL/@stationName"/>
					<xsl:with-param name="CtrlPointH" select="$InstSetupL/l:InstrumentPoint/@pntRef"/>
				</xsl:call-template>
		
		<!-- Reset the TPS Corr Status     -->	<xsl:value-of select="js:ResetTPSCorrStatus()"/>
		
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
<!--	Individual TPS Setup Exports																	-->
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->		
		
	<xsl:choose>
	
	<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	-->
	<!--	TPS Setup - Set Orientation																	-->
	<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	-->
		<xsl:when test="$InstSetupH/@tpsSetupMethod = 'setOrientation'">
		
		<!-- Station Block -->	<xsl:value-of select="concat('//', $returnSep, '//', $tabSep, $tabSep,'Station Information', $returnSep)"/>
		<!-- PTS Line    -->	<xsl:if test="$ExportData = 'ENH'">//</xsl:if>
		<!-- Line Start   -->	<xsl:value-of select="concat(' 3', $tabSep)"/>
		<!-- Export Time  -->	<xsl:if test="$ExportDT = 'y'">
									<xsl:value-of select="concat($auDate,$spaceSep,$auTime)"/>
								</xsl:if>
								<xsl:value-of select="$tabSep"/>
		<!-- Export Code  -->	<xsl:choose>
									<xsl:when test="$FxdCodeStn = 'y'">
										<xsl:value-of select="$FxdCodeStnC"/>
									</xsl:when>
									<xsl:otherwise>
										<xsl:value-of select="$PointH/h:PointCode/@code"/>
									</xsl:otherwise>
								</xsl:choose>
								<xsl:value-of select="concat($tabSep, $tabSep)"/>
		<!-- Station ID   -->	<xsl:value-of select="concat($InstSetupL/@stationName, $tabSep,$InstSetupL/@stationName, $tabSep, $tabSep,format-number($InstSetupL/@instrumentHeight, $dec3), $returnSep)"/>
		
		<!-- Backsight Target Observations -->
			<!-- Observations Header -->
				<xsl:value-of select="concat('//', $tabSep, $tabSep, $tabSep,'Backsight Observation', $returnSep)"/>
			
			<!-- Obs Export -->
				<xsl:call-template name="BsObs">
					<xsl:with-param name="CInstSetup" select="$InstSetupL/@id"/>
					<xsl:with-param name="CPurpose" select="'backsight'"/>
					<xsl:with-param name="ObsType" select="' 4'"/>
				</xsl:call-template>
			
		</xsl:when>
		
	<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	-->
	<!--	TPS Setup - Known Backsight																	-->
	<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	-->
		<xsl:when test="$InstSetupH/@tpsSetupMethod = 'knownBacksight'">
			
			<!-- Coords Export   -->
				<xsl:call-template name="BsCoords">
					<xsl:with-param name="CInstSetup" select="$InstSetupL/@id"/>
					<xsl:with-param name="CPurpose" select="'backsight'"/>
					<xsl:with-param name="RemObs" select="''"/>
				</xsl:call-template>
				
			<!-- Station Block -->
				<!-- Header       -->	<xsl:value-of select="concat('//', $returnSep, '//', $tabSep, $tabSep,'Station Information', $returnSep)"/>
				<!-- PTS Line     -->	<xsl:if test="$ExportData = 'ENH'">//</xsl:if>
				<!-- Line Start   -->	<xsl:value-of select="concat(' 3', $tabSep)"/>
				<!-- Export Time  -->	<xsl:if test="$ExportDT = 'y'">
											<xsl:value-of select="concat($auDate,$spaceSep,$auTime)"/>
										</xsl:if>
										<xsl:value-of select="$tabSep"/>
				<!-- Export Code  -->	<xsl:choose>
											<xsl:when test="$FxdCodeStn = 'y'">
												<xsl:value-of select="$FxdCodeStnC"/>
											</xsl:when>
											<xsl:otherwise>
												<xsl:value-of select="$PointH/h:PointCode/@code"/>
											</xsl:otherwise>
										</xsl:choose>
										<xsl:value-of select="concat($tabSep, $tabSep)"/>
				<!-- Station ID   -->	<xsl:value-of select="concat($InstSetupL/@stationName, $tabSep,$InstSetupL/@stationName, $tabSep, $tabSep,format-number($InstSetupL/@instrumentHeight, $dec3), $returnSep)"/>
			
			<!-- Backsight Target Observations -->
				<!-- Observations Header -->
					<xsl:value-of select="concat('//', $returnSep, '//', $tabSep, $tabSep,'Backsight Observation', $returnSep)"/>
					
				<!-- Obs Export -->
					<xsl:call-template name="BsObs">
						<xsl:with-param name="CInstSetup" select="$InstSetupL/@id"/>
						<xsl:with-param name="CPurpose" select="'backsight'"/>
						<xsl:with-param name="ObsType" select="' 4'"/>
					</xsl:call-template>
				
				<!-- Residuals Export -->
					<xsl:value-of select="concat('//', $returnSep, '//', $tabSep, $tabSep,'Backsight Residual', $returnSep)"/>
					
					<!-- Line Start  -->	<xsl:value-of select="concat('//', $tabSep, $tabSep, $tabSep, $tabSep, $tabSep, $tabSep)"/>
					<!-- Delta Hd    -->	<xsl:value-of select="concat('dHd(m): ', format-number($InstSetupH/h:TPSSetupResult/h:KnownBacksightResults/@deltaHzDist, $dec3), $tabSep)"/>
					<!-- Delta Ht    -->	<xsl:value-of select="concat('dHt(m): ', format-number($InstSetupH/h:TPSSetupResult/h:KnownBacksightResults/@deltaHgt, $dec3))"/>
					<!-- CR/LF       -->	<xsl:value-of select="$returnSep"/>
			
		</xsl:when>	
		
	<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	-->
	<!--	TPS Setup - Resection																		-->
	<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	-->
		<xsl:when test="$InstSetupH/@tpsSetupMethod = 'resection'">
			
			<!-- Coords Export   -->	
				<xsl:call-template name="BsCoords">
					<xsl:with-param name="CInstSetup" select="$InstSetupL/@id"/>
					<xsl:with-param name="CPurpose" select="'resection'"/>
				</xsl:call-template>
						
			<!-- Station Block -->
				<!-- Header       -->	<xsl:value-of select="concat('//', $returnSep, '//', $tabSep, $tabSep,'Station Information', $returnSep)"/>
				<xsl:choose>
					<xsl:when test="$RxnRecalc = 'n'">
						<!-- PTS Line     -->	<xsl:if test="$ExportData = 'ENH'">//</xsl:if>
						<!-- Line Start   -->	<xsl:value-of select="concat(' 3', $tabSep)"/>
						<!-- Export Time  -->	<xsl:if test="$ExportDT = 'y'">
													<xsl:value-of select="concat($auDate,$spaceSep,$auTime)"/>
												</xsl:if>
												<xsl:value-of select="$tabSep"/>
						<!-- Export Code  -->	<xsl:choose>
													<xsl:when test="$FxdCodeStn = 'y'">
														<xsl:value-of select="$FxdCodeStnC"/>
													</xsl:when>
													<xsl:otherwise>
														<xsl:value-of select="$PointH/h:PointCode/@code"/>
													</xsl:otherwise>
												</xsl:choose>
												<xsl:value-of select="concat($tabSep, $tabSep)"/>
						<!-- Station ID   -->	<xsl:value-of select="concat($InstSetupL/@stationName, $tabSep,$InstSetupL/@stationName, $tabSep, $tabSep,format-number($InstSetupL/@instrumentHeight, $dec3), $returnSep)"/>
					</xsl:when>
					<xsl:otherwise>
						<!-- Line Start   -->	<xsl:value-of select="concat('128', $tabSep, $tabSep,$PointH/h:PointCode/@code, $tabSep, $tabSep)"/>
						<!-- Station ID   -->	<xsl:value-of select="concat($InstSetupL/@stationName, $tabSep,$InstSetupL/@stationName, $tabSep, $tabSep,format-number($InstSetupL/@instrumentHeight, $dec3), $returnSep)"/>
					</xsl:otherwise>
				</xsl:choose>
			
			<!-- Backsight Target Observations -->
				<!-- Observations Header -->
					<xsl:value-of select="concat('//', $returnSep, '//', $tabSep, $tabSep,'Resection Observations', $returnSep)"/>
					
				<!-- Obs Export -->	
					<xsl:call-template name="BsObs">
						<xsl:with-param name="CInstSetup" select="$InstSetupL/@id"/>
						<xsl:with-param name="CPurpose" select="'resection'"/>
						<xsl:with-param name="ObsType" select="' 7'"/>
					</xsl:call-template>
			
			<!-- Resection ReCalc End -->
				<xsl:if test="$RxnRecalc = 'y'">
					<xsl:value-of select="concat('129', $returnSep)"/>
				</xsl:if>
				
			<!-- Resection Target Residuals -->
					
				<!-- Residual Header     -->	<xsl:value-of select="concat('//', $returnSep, '//', $tabSep, $tabSep,'Individual Residuals', $returnSep)"/>
					
				<!-- Residuals Export -->	
					<xsl:call-template name="BsResiduals">
						<xsl:with-param name="CInstSetup" select="$InstSetupL/@id"/>
					</xsl:call-template>
			
		</xsl:when>	
		
	<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	-->
	<!--	TPS Setup - Multiple Backsights																-->
	<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	-->
		<xsl:when test="$InstSetupH/@tpsSetupMethod = 'multipleBacksights'">
			
			<!-- Coords Export   -->	
				<xsl:call-template name="BsCoords">
					<xsl:with-param name="CInstSetup" select="$InstSetupL/@id"/>
					<xsl:with-param name="CPurpose" select="'backsight'"/>
				</xsl:call-template>
				
			<!-- Station Block -->
				<!-- Header       -->	<xsl:value-of select="concat('//', $returnSep, '//', $tabSep, $tabSep,'Station Information', $returnSep)"/>
				<!-- PTS Line     -->	<xsl:if test="$ExportData = 'ENH'">//</xsl:if>
				<!-- Line Start   -->	<xsl:value-of select="concat(' 3', $tabSep)"/>
				<!-- Export Time  -->	<xsl:if test="$ExportDT = 'y'">
											<xsl:value-of select="concat($auDate,$spaceSep,$auTime)"/>
										</xsl:if>
										<xsl:value-of select="$tabSep"/>
				<!-- Export Code  -->	<xsl:choose>
											<xsl:when test="$FxdCodeStn = 'y'">
												<xsl:value-of select="$FxdCodeStnC"/>
											</xsl:when>
											<xsl:otherwise>
												<xsl:value-of select="$PointH/h:PointCode/@code"/>
											</xsl:otherwise>
										</xsl:choose>
										<xsl:value-of select="concat($tabSep, $tabSep)"/>
				<!-- Station ID   -->	<xsl:value-of select="concat($InstSetupL/@stationName, $tabSep,$InstSetupL/@stationName, $tabSep, $tabSep,format-number($InstSetupL/@instrumentHeight, $dec3), $returnSep)"/>
			
			<!-- Backsight Target Observations -->
				<!-- Observations Header -->
					<xsl:value-of select="concat('//', $returnSep, '//', $tabSep, $tabSep,'Backsight Observations', $returnSep)"/>
				
				<!-- Obs Export -->	
					<xsl:call-template name="BsObs">
						<xsl:with-param name="CInstSetup" select="$InstSetupL/@id"/>
						<xsl:with-param name="CPurpose" select="'backsight'"/>
						<xsl:with-param name="ObsType" select="' 7'"/>
					</xsl:call-template>
			
			<!-- Resection Target Residuals -->
					
				<!-- Residual Header     -->	<xsl:value-of select="concat('//', $returnSep, '//', $tabSep, $tabSep,'Individual Residuals', $returnSep)"/>
				
				<!-- Residuals Export -->	
					<xsl:call-template name="BsResiduals">
						<xsl:with-param name="CInstSetup" select="$InstSetupL/@id"/>
					</xsl:call-template>
			
		</xsl:when>	
	
	<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	-->
	<!--	TPS Setup - Height Transfer																	-->
	<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	-->
		<xsl:when test="$InstSetupH/@tpsSetupMethod = 'heightTransfer'">
			
			<!-- Coords Export   -->	
				<xsl:call-template name="BsCoords">
					<xsl:with-param name="CInstSetup" select="$InstSetupL/@id"/>
					<xsl:with-param name="CPurpose" select="'resection'"/>
				</xsl:call-template>
				
			<!-- Station Block -->
				<!-- Header       -->	<xsl:value-of select="concat('//', $returnSep, '//', $tabSep, $tabSep,'Station Information', $returnSep)"/>
				<!-- PTS Line     -->	<xsl:if test="$ExportData = 'ENH'">//</xsl:if>
				<!-- Line Start   -->	<xsl:value-of select="concat(' 3', $tabSep)"/>
				<!-- Export Time  -->	<xsl:if test="$ExportDT = 'y'">
											<xsl:value-of select="concat($auDate,$spaceSep,$auTime)"/>
										</xsl:if>
										<xsl:value-of select="$tabSep"/>
				<!-- Export Code  -->	<xsl:choose>
											<xsl:when test="$FxdCodeStn = 'y'">
												<xsl:value-of select="$FxdCodeStnC"/>
											</xsl:when>
											<xsl:otherwise>
												<xsl:value-of select="$PointH/h:PointCode/@code"/>
											</xsl:otherwise>
										</xsl:choose>
										<xsl:value-of select="concat($tabSep, $tabSep)"/>
				<!-- Station ID   -->	<xsl:value-of select="concat($InstSetupL/@stationName, $tabSep,$InstSetupL/@stationName, $tabSep, $tabSep,format-number($InstSetupL/@instrumentHeight, $dec3), $returnSep)"/>
			
			<!-- Backsight Target Observations -->
				<!-- Observations Header -->
					<xsl:value-of select="concat('//', $returnSep, '//', $tabSep, $tabSep,'Resection Observations', $returnSep)"/>
					
				<!-- Obs Export -->	
					<xsl:call-template name="BsObs">
						<xsl:with-param name="CInstSetup" select="$InstSetupL/@id"/>
						<xsl:with-param name="CPurpose" select="'resection'"/>
						<xsl:with-param name="ObsType" select="' 7'"/>
					</xsl:call-template>
			
			<!-- Resection Target Residuals -->
					
				<!-- Residual Header     -->	<xsl:value-of select="concat('//', $returnSep, '//', $tabSep, $tabSep,'Individual Residuals', $returnSep)"/>
					
				<!-- Residuals Export -->	
					<xsl:call-template name="BsResiduals">
						<xsl:with-param name="CInstSetup" select="$InstSetupL/@id"/>
					</xsl:call-template>
			
		</xsl:when>
		
	<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	-->
	<!--	TPS Setup - Orientate to Line																-->
	<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	-->
		<xsl:when test="$InstSetupH/@tpsSetupMethod = 'orientateLine'">
			
			<!-- Coords Export   -->	
				<xsl:call-template name="BsCoords">
					<xsl:with-param name="CInstSetup" select="$InstSetupL/@id"/>
					<xsl:with-param name="CPurpose" select="'resection'"/>
				</xsl:call-template>
				
			<!-- Station Block -->
				<!-- Header       -->	<xsl:value-of select="concat('//', $returnSep, '//', $tabSep, $tabSep,'Station Information', $returnSep)"/>
				<!-- PTS Line     -->	<xsl:if test="$ExportData = 'ENH'">//</xsl:if>
				<!-- Line Start   -->	<xsl:value-of select="concat(' 3', $tabSep)"/>
				<!-- Export Time  -->	<xsl:if test="$ExportDT = 'y'">
											<xsl:value-of select="concat($auDate,$spaceSep,$auTime)"/>
										</xsl:if>
										<xsl:value-of select="$tabSep"/>
				<!-- Export Code  -->	<xsl:choose>
											<xsl:when test="$FxdCodeStn = 'y'">
												<xsl:value-of select="$FxdCodeStnC"/>
											</xsl:when>
											<xsl:otherwise>
												<xsl:value-of select="$PointH/h:PointCode/@code"/>
											</xsl:otherwise>
										</xsl:choose>
										<xsl:value-of select="concat($tabSep, $tabSep)"/>
				<!-- Station ID   -->	<xsl:value-of select="concat($InstSetupL/@stationName, $tabSep,$InstSetupL/@stationName, $tabSep, $tabSep,format-number($InstSetupL/@instrumentHeight, $dec3), $returnSep)"/>
			
			<!-- Backsight Target Observations -->
				<!-- Observations Header -->
					<xsl:value-of select="concat('//', $returnSep, '//', $tabSep, $tabSep,'Resection Observations', $returnSep)"/>
					
				<!-- Obs Export -->	
					<xsl:call-template name="BsObs">
						<xsl:with-param name="CInstSetup" select="$InstSetupL/@id"/>
						<xsl:with-param name="CPurpose" select="'resection'"/>
						<xsl:with-param name="ObsType" select="' 7'"/>
					</xsl:call-template>
			
			<!-- Resection Target Residuals -->
					
				<!-- Residual Header     -->	<xsl:value-of select="concat('//', $returnSep, '//', $tabSep, $tabSep,'Individual Residuals', $returnSep)"/>
					
				<!-- Residuals Export -->	
					<xsl:call-template name="BsResiduals">
						<xsl:with-param name="CInstSetup" select="$InstSetupL/@id"/>
					</xsl:call-template>
			
		</xsl:when>	

	<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	-->
	<!--	TPS Setup - Orientate to Object																-->
	<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	-->
		<xsl:when test="$InstSetupH/@tpsSetupMethod = 'orientateObject'">
			
			<!-- Coords Export   -->	
				<xsl:call-template name="BsCoords">
					<xsl:with-param name="CInstSetup" select="$InstSetupL/@id"/>
					<xsl:with-param name="CPurpose" select="'resection'"/>
				</xsl:call-template>
				
			<!-- Station Block -->
				<!-- Header       -->	<xsl:value-of select="concat('//', $returnSep, '//', $tabSep, $tabSep,'Station Information', $returnSep)"/>
				<!-- PTS Line     -->	<xsl:if test="$ExportData = 'ENH'">//</xsl:if>
				<!-- Line Start   -->	<xsl:value-of select="concat(' 3', $tabSep)"/>
				<!-- Export Time  -->	<xsl:if test="$ExportDT = 'y'">
											<xsl:value-of select="concat($auDate,$spaceSep,$auTime)"/>
										</xsl:if>
										<xsl:value-of select="$tabSep"/>
				<!-- Export Code  -->	<xsl:choose>
											<xsl:when test="$FxdCodeStn = 'y'">
												<xsl:value-of select="$FxdCodeStnC"/>
											</xsl:when>
											<xsl:otherwise>
												<xsl:value-of select="$PointH/h:PointCode/@code"/>
											</xsl:otherwise>
										</xsl:choose>
										<xsl:value-of select="concat($tabSep, $tabSep)"/>
				<!-- Station ID   -->	<xsl:value-of select="concat($InstSetupL/@stationName, $tabSep,$InstSetupL/@stationName, $tabSep, $tabSep,format-number($InstSetupL/@instrumentHeight, $dec3), $returnSep)"/>
			
			<!-- Backsight Target Observations -->
				<!-- Observations Header -->
					<xsl:value-of select="concat('//', $returnSep, '//', $tabSep, $tabSep,'Resection Observations', $returnSep)"/>
					
				<!-- Obs Export -->	
					<xsl:call-template name="BsObs">
						<xsl:with-param name="CInstSetup" select="$InstSetupL/@id"/>
						<xsl:with-param name="CPurpose" select="'resection'"/>
						<xsl:with-param name="ObsType" select="' 7'"/>
					</xsl:call-template>
			
			<!-- Resection Target Residuals -->
					
				<!-- Residual Header     -->	<xsl:value-of select="concat('//', $returnSep, '//', $tabSep, $tabSep,'Individual Residuals', $returnSep)"/>
					
				<!-- Residuals Export -->	
					<xsl:call-template name="BsResiduals">
						<xsl:with-param name="CInstSetup" select="$InstSetupL/@id"/>
					</xsl:call-template>
		</xsl:when>
		
	</xsl:choose>
	
		<xsl:value-of select="concat('//',$returnSep)"/>
		<xsl:value-of select="$divline"/>
		
		<!-- Reset the TargetHt to default --> <xsl:value-of select="js:ResetCurrentTH(number($defaultHT))"/>
		<!-- Reset the EDMATR to default   --> <xsl:value-of select="js:ResetEDMATR()"/>
		
	</xsl:template>

<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
<!--	Control point Export																			-->
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
	<xsl:template name="CtrlExport">
		<xsl:param name="CtrlPnt"/>
		<xsl:param name="CtrlPointH"/>
		<xsl:variable name="PointH" select="key('KeyPointH',$CtrlPointH)"/>
		<xsl:variable name="CgPoint" select="key('KeyCgPointL',$CtrlPointH)"/>
		
		<!-- Line Begin   -->	<xsl:value-of select="concat(' 2', $tabSep)"/>
		<!-- Export Time  -->	<xsl:if test="$ExportDT = 'y'">
									<xsl:value-of select="concat(js:XMLTimeAuDate(string($CgPoint/@timeStamp)),$spaceSep,js:XMLTimeAuTime(string($CgPoint/@timeStamp)))"/>
								</xsl:if>
								<xsl:value-of select="$tabSep"/>
		<!-- Export Code  -->	<xsl:choose>
									<xsl:when test="$FxdCodeCtrl = 'y'">
										<xsl:value-of select="$FxdCodeCtrlC"/>
									</xsl:when>
									<xsl:otherwise>
										<xsl:value-of select="$PointH/h:PointCode/@code"/>
									</xsl:otherwise>
								</xsl:choose>
								<xsl:value-of select="concat($tabSep, $tabSep)"/>
		<!-- Station Name -->	<xsl:value-of select="concat($CtrlPnt, $tabSep, $CtrlPnt, $tabSep)"/>
		<!-- Attributes   -->	<xsl:if test="$PointH/h:PointCode/h:Attribute">
									<xsl:value-of select="$PointH/h:PointCode/h:Attribute/@value"/>
								</xsl:if>
		<!-- Tab          -->	<xsl:value-of select="$tabSep"/>
		<!-- Easting      -->	<xsl:value-of select="concat(format-number($PointH/h:Coordinates/h:Local/h:Grid/@e, $dec3), $tabSep)"/>
		<!-- Northing     -->	<xsl:value-of select="concat(format-number($PointH/h:Coordinates/h:Local/h:Grid/@n, $dec3), $tabSep)"/>								
		<!-- Height       -->	<xsl:value-of select="format-number($PointH/h:Coordinates/h:Local/h:Grid/@hghthO, $dec3)"/>
		<!-- CR/LF        -->	<xsl:value-of select="$returnSep"/>
		
	</xsl:template>

<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
<!--	BS Point Residuals																				-->
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
	<xsl:template name="BsResiduals">
		<xsl:param name="CInstSetup"/>
		
		<xsl:variable name="InstSetupH" select="key('KeyInstSetupH',$CInstSetup)"/>
		
		<xsl:for-each select="$InstSetupH/h:TPSSetupResult/*/h:BacksightResults">
			<xsl:sort select="key('KeyCgPointL', @id)/@timeStamp" order="ascending"/>
			<xsl:variable name="BacksightResults" select="."/>
			<xsl:variable name="CgPointL" select="key('KeyCgPointL', $BacksightResults/@id)"/>
			<!-- Line Start  -->	<xsl:value-of select="concat('//', $tabSep, $tabSep, $tabSep, $tabSep)"/>
			<!-- Point ID    -->	<xsl:value-of select="concat($CgPointL/@oID, $tabSep)"/>
			<!-- Use Flag    -->	<xsl:value-of select="concat('Used: ', $BacksightResults/@useKind, $tabSep)"/>
			<!-- Delta Hz    -->	<xsl:value-of select="concat('dHz(dms): ', format-number($BacksightResults/@deltaHz, $dec4), $tabSep)"/>
			<!-- Delta Hd    -->	<xsl:value-of select="concat('dHd(m): ', format-number($BacksightResults/@deltaHzDist, $dec3), $tabSep)"/>
			<!-- Delta Ht    -->	<xsl:value-of select="concat('dHt(m): ', format-number($BacksightResults/@deltaHgt, $dec3))"/>
			<!-- CR/LF       -->	<xsl:value-of select="$returnSep"/>
		</xsl:for-each>
	</xsl:template>
	
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
<!--	BS Point Observations																			-->
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
	<xsl:template name="BsObs">
		<xsl:param name="CInstSetup"/>
		<xsl:param name="CPurpose"/>
		<xsl:param name="ObsType"/>
	
		<xsl:for-each select="//l:RawObservation[@setupID=$CInstSetup and @purpose=$CPurpose]">
			<xsl:sort select="@timeStamp" order="ascending"/>
			<xsl:variable name="RawObs" select="."/>
			<xsl:variable name="BSPointH" select="key('KeyPointH', $RawObs/l:TargetPoint/@pntRef)"/>
			
			<xsl:call-template name="ObsExport">
				<xsl:with-param name="CgPointName" select="$RawObs/l:TargetPoint/@pntRef"/>
				<xsl:with-param name="SurveyObs" select="'n'"/>
				<xsl:with-param name="ObsType" select="$ObsType"/>
			</xsl:call-template>
			
		</xsl:for-each>
	</xsl:template>

<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
<!--	BS Point Coordinates																			-->
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
	<xsl:template name="BsCoords">
		<xsl:param name="CInstSetup"/>
		<xsl:param name="CPurpose"/>
	
		<xsl:for-each select="//l:RawObservation[@setupID=$CInstSetup and @purpose=$CPurpose]">
			<xsl:sort select="@timeStamp" order="ascending"/>
			<xsl:variable name="RawObs" select="."/>
			<xsl:variable name="BSPointH" select="key('KeyPointH', $RawObs/l:TargetPoint/@pntRef)"/>
			
			<xsl:call-template name="CtrlExport">
				<xsl:with-param name="CtrlPnt" select="$RawObs/l:TargetPoint/@name"/>
				<xsl:with-param name="CtrlPointH" select="$RawObs/l:TargetPoint/@pntRef"/>
			</xsl:call-template>
			
		</xsl:for-each>
	</xsl:template>

<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
<!--	Code Attributes Export																			-->
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
	<xsl:template name="CodeAtt">
		<xsl:param name="PointH"/>
		
		<xsl:variable name="CodeH" select="key('KeyCodeH',$PointH/h:PointCode/@code)"/>
		
	<xsl:value-of select="concat('124', $tabSep, $tabSep, $tabSep, 'Survey Attributes', $tabSep, '0', $returnSep)"/>
	<xsl:for-each select="$PointH/h:PointCode/h:Attribute">
		<xsl:variable name="attPointCode" select="."/>
		<xsl:if test="$attPointCode/@name != $AttString">
			<xsl:choose>
				<xsl:when test="$attPointCode/@name = 'Tinnable'">
					<xsl:value-of select="concat('71', $tabSep, $tabSep, 'Tinnable', $tabSep)"/>
					<xsl:value-of select="concat($attPointCode/@value, $returnSep)"/>
					<xsl:value-of select="js:TinnableFlag(number($attPointCode/@value))"/>
				</xsl:when>												
				<xsl:otherwise>
				
					<xsl:for-each select="$CodeH/h:PointCode/h:CodeAttributes">
						<xsl:if test="@attributeName = $attPointCode/@name">
							<xsl:choose>
								<xsl:when test="@valueType = 'Integer'">
									<xsl:value-of select="concat('71', $tabSep, $tabSep, $attPointCode/@name, $tabSep, $attPointCode/@value, $returnSep)"/>
								</xsl:when>
								<xsl:when test="@valueType = 'Real'">
									<xsl:value-of select="concat('72', $tabSep, $tabSep, $attPointCode/@name, $tabSep, format-number($attPointCode/@value,$dec3), $returnSep)"/>
								</xsl:when>
								<xsl:otherwise>
									<xsl:value-of select="concat('73', $tabSep, $tabSep, $attPointCode/@name, $tabSep, $attPointCode/@value, $returnSep)"/>
								</xsl:otherwise>
							</xsl:choose>
						</xsl:if>
					</xsl:for-each>
					
				</xsl:otherwise>
			</xsl:choose>
		</xsl:if>
	</xsl:for-each>
	<xsl:value-of select="concat('125', $tabSep, $tabSep, $tabSep, $tabSep, '0', $returnSep)"/>
	</xsl:template>

<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
<!--	TPS & GPS Obsservation Code & Attributes Export														-->
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
	<xsl:template name="ObsAtt">
		<xsl:param name="PointH"/>
		<xsl:param name="CgPointName"/>
		<!--<xsl:variable name="CodeH" select="key('KeyCodeH',$PointH/h:PointCode/@code)"/> -->
		<xsl:variable name="CgPoint" select ="key('KeyCgPointL', $CgPointName)"/>
		<xsl:variable name="InstSetupL" select ="key('KeyInstSetupL', $CgPointName)"/>
		<xsl:variable name="RawObsL" select ="key('KeyRawObsL', $CgPointName)"/>
		<xsl:variable name="RawObsH" select ="key('KeyRawObsH', $CgPointName)"/>
		<xsl:variable name="GPSPosL" select ="key('KeyGPSPosL', $CgPointName)"/>
		<xsl:variable name="GPSPosH" select="key('KeyGPSPosH', $CgPointName)"/>
		<xsl:variable name="GPSPointH" select="key('KeyPointH', $CgPointName)"/>
		<xsl:variable name="GPSRefH" select ="key('KeyPointH', $GPSPosH/h:RTKInfo/@referenceRef)"/>
		
		<xsl:choose>
			<!-- TPS Observation Data -->
			<xsl:when test="$PointH/@subclass = 'TPS'">
				<!-- Start-TPS   -->	<xsl:value-of select="concat('124', $tabSep, $tabSep, $tabSep, 'Leica TPS EDM Information', $tabSep, '0', $returnSep)"/>
				
				<!-- Date        -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Date', $tabSep,js:XMLTimeAuDate(string($CgPoint/@timeStamp)), $returnSep)"/>
				<!-- Time        -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Time', $tabSep,js:XMLTimeAuTime(string($CgPoint/@timeStamp)), $returnSep)"/>
				<!-- TPS Station -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'TPS Reference', $tabSep,$InstSetupL/@stationName, $returnSep)"/>			
				<!-- EDM Type    -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'EDM Type', $tabSep, $RawObsH/@edmKind, $returnSep)"/>
				<!-- ATR Mode    -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'ATR Mode', $tabSep, $RawObsH/@atrMode, $returnSep)"/>	
				<!-- Tilt Comp.  -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Tilt Compensated', $tabSep, $RawObsH/@tiltCompensated, $returnSep)"/>
				<!-- Ht Comp.    -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Height Compensated', $tabSep, $RawObsH/@autoHeight, $returnSep)"/>
				<!-- Reflector Ht-->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Reflector Height', $tabSep, format-number($RawObsL/@targetHeight, $dec3), $returnSep)"/>			
				<!-- Prism Type  -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Reflector Type', $tabSep, $RawObsH/@reflectorName, $returnSep)"/>
				<!-- Prism Const -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Reflector Constant', $tabSep, format-number($RawObsH/@reflectorConstant,$dec4), $returnSep)"/>
				<!-- Offset Check-->	<xsl:if test="$RawObsL/l:OffsetVals">
					<!-- OS I/O  -->		<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Offset Length', $tabSep, format-number($RawObsL/l:OffsetVals/@offsetInOut,$dec3), $returnSep)"/>
					<!-- OS L/R  -->		<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Offset Cross', $tabSep, format-number($RawObsL/l:OffsetVals/@offsetLeftRight,$dec3), $returnSep)"/>
					<!-- OS U/D  -->		<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Offset Height', $tabSep, format-number($RawObsL/l:OffsetVals/@offsetUpDown,$dec3), $returnSep)"/>
										</xsl:if>
				<!-- End-TPS     -->	<xsl:value-of select="concat('125', $tabSep, $tabSep, $tabSep, $tabSep, '0', $returnSep)"/>
			</xsl:when>
			
			<!-- GNSS Observation Data -->
			<xsl:otherwise>
				<!-- Exclude if GPS Hidden point -->
				<xsl:if test="$PointH/@subclass != 'GPS hidden point'">
					
					<!-- Start-GPS       -->	<xsl:value-of select="concat('124', $tabSep, $tabSep, $tabSep, 'Leica GNSS Information', $tabSep, '0', $returnSep)"/>
					<!-- Date            -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Date', $tabSep,js:XMLTimeAuDate(string($CgPoint/@timeStamp)), $returnSep)"/>
					<!-- Time            -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Time', $tabSep,js:XMLTimeAuTime(string($CgPoint/@timeStamp)), $returnSep)"/>
					<!-- Antenna Height  -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Antenna Height', $tabSep, format-number($GPSPosL/../@antennaHeight, $dec3), $returnSep)"/>
					<!-- GNSS Solution   -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'GNSS Solution', $tabSep,$PointH/@subclass, $returnSep)"/>
					<!-- RTK Positions   -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'RTK Positions', $tabSep, $GPSPosH/h:RTKInfo/@numRTKPositionUsed, $returnSep)"/>
					<!-- Tilt Comp.      -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Tilt Compensated', $tabSep, $GPSPosH/@tiltCompensated, $returnSep)"/>
					<!-- Tilt Data       -->	<xsl:if test="$GPSPosH/@tiltCompensated = 'true'">
					<!-- Tilt Direct WGS -->		<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Tilt Direction (WGS84)', $tabSep, js:DDtoDMSTxt(string(js:ConvertDMStoDD(string($GPSPosH/h:TiltInfo/@tiltDirection))),5), $returnSep)"/>
					<!-- Tilt Value      -->		<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Tilt', $tabSep, js:DDtoDMSTxt(string(js:ConvertDMStoDD(string($GPSPosH/h:TiltInfo/@tilt))),0), $returnSep)"/>
												</xsl:if>
					<!-- PDOP            -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'PDOP', $tabSep,format-number($GPSPosH/h:GPSQuality/@PDOP, $dec3), $returnSep)"/>
					<!-- 3D CQ           -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, '3D Quality', $tabSep,format-number($PointH/h:PointQuality/@CQ3D, $dec3), $returnSep)"/>
					<!-- 2D CQ           -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Position Quality', $tabSep,format-number($PointH/h:PointQuality/@CQPos, $dec3), $returnSep)"/>
					<!-- 1D CQ           -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Height Quality', $tabSep,format-number($PointH/h:PointQuality/@CQHeight, $dec3), $returnSep)"/>
					<!-- PU Export       -->	<xsl:if test="$PUexport = 'y'">
													<!-- PU 95% Position -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, '2D PU 95 Pcnt cl', $tabSep,format-number(js:PU95cl(number($PointH/h:PointQuality/@CQPos),$BasePcl), $dec3), $returnSep)"/>
													<!-- PU 95% Height   -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, '1D PU 95 Pcnt cl', $tabSep,format-number(js:PU95cl(number($PointH/h:PointQuality/@CQHeight),$BaseHcl), $dec3), $returnSep)"/>
												</xsl:if>
												
					<!-- Total SV used   -->	<xsl:variable name="totalSVtracked" select="($GPSPosH/h:SatelliteInfo/@GPSSatUsed + $GPSPosH/h:SatelliteInfo/@GLONASSSatUsed + $GPSPosH/h:SatelliteInfo/@BEIDOUSatUsed + $GPSPosH/h:SatelliteInfo/@GALILEOSatUsed)"/>
												<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Total Sats Used', $tabSep, $totalSVtracked, $returnSep)"/>				
					<!-- GPS Sats        -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Sats GPS Used Trk', $tabSep, $GPSPosH/h:SatelliteInfo/@GPSSatUsed, ' ',$GPSPosH/h:SatelliteInfo/@GPSSatTracked, $returnSep)"/>
					<!-- GLO Sats        -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Sats GLO Used Trk', $tabSep, $GPSPosH/h:SatelliteInfo/@GLONASSSatUsed, ' ',$GPSPosH/h:SatelliteInfo/@GLONASSSatTracked, $returnSep)"/>
					<!-- BDS Sats        -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Sats BDS Used Trk', $tabSep, $GPSPosH/h:SatelliteInfo/@BEIDOUSatUsed, ' ',$GPSPosH/h:SatelliteInfo/@BEIDOUSatTracked, $returnSep)"/>	
					<!-- GAL Sats        -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Sats GAL Used Trk', $tabSep, $GPSPosH/h:SatelliteInfo/@GALILEOSatUsed, ' ',$GPSPosH/h:SatelliteInfo/@GALILEOSatTracked, $returnSep)"/>	
					<!-- End-GPS         -->	<xsl:value-of select="concat('125', $tabSep, $tabSep, $tabSep, $tabSep, '0', $returnSep)"/>
					
					<xsl:if test="$ExportRef ='y'">
						<!-- Start-Ref       -->	<xsl:value-of select="concat('124', $tabSep, $tabSep, $tabSep, 'GNSS Reference Information', $tabSep, '0', $returnSep)"/>
						<!-- Reference ID    -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Reference ID', $tabSep, $GPSPosH/h:RTKInfo/@referenceRef, $returnSep)"/>
						<!-- Easting         -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Easting', $tabSep, format-number($GPSRefH/h:Coordinates/h:Local/h:Grid/@e,$dec3), $returnSep)"/>
						<!-- Northing        -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Northing', $tabSep, format-number($GPSRefH/h:Coordinates/h:Local/h:Grid/@n,$dec3), $returnSep)"/>
						<!-- Orth Height     -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Orthometric Height', $tabSep, format-number($GPSRefH/h:Coordinates/h:Local/h:Grid/@hghthO,$dec3), $returnSep)"/>
						<!-- Latitude        -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Latitude', $tabSep, js:DDtoDMSTxt(string($GPSRefH/h:Coordinates/h:Local/h:Geodetic/@lat),5), $returnSep)"/>
						<!-- Longitude       -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Longitude', $tabSep, js:DDtoDMSTxt(string($GPSRefH/h:Coordinates/h:Local/h:Geodetic/@lon),5), $returnSep)"/>
						<!-- Ellipsoidal Ht  -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Ellipsoidal Height', $tabSep, format-number($GPSRefH/h:Coordinates/h:Local/h:Geodetic/@hghtE,$dec3), $returnSep)"/>
						<!-- PU Export       -->	<xsl:if test="$PUexport = 'y'">
														<!-- PU 95% Position -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'CORS 2D PU 95 Pcnt cl', $tabSep,format-number($BasePcl, $dec3), $returnSep)"/>
														<!-- PU 95% Height   -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'CORS 1D PU 95 Pcnt cl', $tabSep,format-number($BaseHcl, $dec3), $returnSep)"/>
													</xsl:if>
						<!-- Network Solution-->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Network Solution', $tabSep, $GPSPosH/h:RTKInfo/@networkSolution, $returnSep)"/>
						<!-- Network Type    -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Network Type', $tabSep, $GPSPosH/h:RTKInfo/@networkType, $returnSep)"/>
						<!-- Inside Network  -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Inside Network', $tabSep, $GPSPosH/h:RTKInfo/@insideRTKNetwork, $returnSep)"/>
						<!-- Server          -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Server Address', $tabSep, $GPSPosH/h:RTKInfo/@ipAddress,$colSep, $GPSPosH/h:RTKInfo/@port, $returnSep)"/>
						<!-- Mount Point     -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'Mount Point', $tabSep, $GPSPosH/h:RTKInfo/@mountpoint, $returnSep)"/>
						<!-- RTK Data Format -->	<xsl:value-of select="concat('73', $tabSep, $tabSep, 'RTK Data Format', $tabSep, $GPSPosH/h:RTKInfo/@dataFormat, $returnSep)"/>
						<!-- End-GPS         -->	<xsl:value-of select="concat('125', $tabSep, $tabSep, $tabSep, $tabSep, '0', $returnSep)"/>
					</xsl:if>
				</xsl:if>
			</xsl:otherwise>
		</xsl:choose>
		
		<!-- Reset the TargetHt to default --> <xsl:value-of select="js:ResetCurrentTH(number($defaultHT))"/>
		<!-- Reset the EDMATR to default   --> <xsl:value-of select="js:ResetEDMATR()"/>
		
	</xsl:template>

<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
<!--	TPS Offsets																						-->
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
	<xsl:template name="TPSoffsets">
		<xsl:param name="CgPointName"/>
		<xsl:variable name="RawObsL" select ="key('KeyRawObsL', $CgPointName)"/>
		
		<xsl:if test="$RawObsL/l:OffsetVals">
			<!-- PTS Line    -->	<xsl:if test="$ExportData = 'ENH'">//</xsl:if>
			<!-- O/S In/Out  -->	<xsl:value-of select="concat('42', $tabSep, $tabSep, format-number($RawObsL/l:OffsetVals/@offsetInOut,$dec3),$returnSep)"/>
			<!-- PTS Line    -->	<xsl:if test="$ExportData = 'ENH'">//</xsl:if>
			<!-- O/S L/R     -->	<xsl:value-of select="concat('43', $tabSep, $tabSep, format-number($RawObsL/l:OffsetVals/@offsetLeftRight,$dec3),$returnSep)"/>
			<!-- PTS Line    -->	<xsl:if test="$ExportData = 'ENH'">//</xsl:if>
			<!-- O/S Up/Dn   -->	<xsl:value-of select="concat('44', $tabSep, $tabSep, format-number($RawObsL/l:OffsetVals/@offsetUpDown,$dec3),$returnSep)"/>
		</xsl:if>
	</xsl:template>
	
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
<!--	Free Codes Export																				-->
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
	<xsl:template match="h:FreeCode">
	
		<xsl:choose>
			<xsl:when test="$ExportFC = 'c'">
				<xsl:value-of select="@code"/>
			</xsl:when>
			<xsl:otherwise>
				<xsl:value-of select="@codeDesc"/>
			</xsl:otherwise>
		</xsl:choose>
		<xsl:if test="h:Attribute">
			<xsl:value-of select="$tabSep"/>
			<xsl:for-each select="h:Attribute">
				<xsl:value-of select="concat($tabSep,@value)"/>
			</xsl:for-each>
		</xsl:if>
		<xsl:value-of select="$returnSep"/>
	</xsl:template>
	
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
<!--	String Number Extract																			-->
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
	<xsl:template name="GetString">
		<xsl:param name="CgPointName"/>
		
		<!-- Linework Array Index    -->
		<xsl:variable name="MasterPointIndex" select="js:getIndex(string($CgPointName))"/>
			
		<xsl:choose>
			<xsl:when test="$Coding = 'CAPTIVATE'">
				<xsl:if test="$MasterPointIndex > -1">
					<xsl:variable name="currentLineId" select="js:getLineId(string($MasterPointIndex))"/>
					<xsl:value-of select="format-number(substring-before(substring-after($currentLineId,'_'),'_'),$StrNo)"/>
				</xsl:if>
				<xsl:if test="$MasterPointIndex = -1">
					<xsl:variable name="AttributeString" select="h:PointCode/h:Attribute[@name=$AttString]"/>
						<xsl:choose>
							<xsl:when test="$AttributeString">
								<xsl:value-of select="format-number($AttributeString/@value,$StrNo)"/>
							</xsl:when>
							<xsl:otherwise>
								<!--<xsl:value-of select=""/>-->
							</xsl:otherwise>
						</xsl:choose>
				</xsl:if>
			</xsl:when>
			<xsl:otherwise>
				<!-- Old Sting Attribute used -->
					<xsl:variable name="AttributeString" select="h:PointCode/h:Attribute[@name=$AttString]"/>
						<xsl:choose>
							<xsl:when test="$AttributeString">
								<xsl:value-of select="format-number($AttributeString/@value,$StrNo)"/>
							</xsl:when>
							<xsl:otherwise>
								<!--<xsl:value-of select=""/>-->
							</xsl:otherwise>
						</xsl:choose>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>

<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
<!--	Close String Extract																			-->
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
	<xsl:template name="GetCloseString">
		<xsl:param name="CgPointName"/>
		
		<!-- Linework Array Index    -->
		<xsl:variable name="MasterPointIndex" select="js:getIndex(string($CgPointName))"/>
		
		<xsl:if test="//l:Application/@name = 'Leica Captivate LandXML Export'">
					<xsl:if test="$MasterPointIndex > -1">
						<xsl:variable name="currentLineWork" select="js:getLineWork(string($MasterPointIndex))"/>
						<xsl:variable name="currentLineStrtEnd" select="js:getLineStrtEnd(string($MasterPointIndex))"/>
						<xsl:variable name="currentLineType" select="js:getLineType(string($MasterPointIndex))"/>
						<xsl:variable name="LineTypeOpCode" select="js:LineworkLookup(string($currentLineType), string($currentLineStrtEnd))"/>
						<xsl:value-of select="$LineTypeOpCode"/>
						<xsl:if test="$currentLineWork = $CLcl">
							<xsl:value-of select="concat('20',$returnSep)"/>
						</xsl:if>
					</xsl:if>
				</xsl:if>
	</xsl:template>
	
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
<!--	Land XML Linework																				-->
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
	<xsl:template name="LineworkExtract">
		<xsl:variable name="CoordGeom" select="."/>
		
		<!-- Spline Start & End Export -->
		<xsl:for-each select="//l:CoordGeom/l:IrregularLine/*">
			<xsl:variable name="LineFeat" select="key('KeyLinesH', ../@oID)"/>
			<xsl:variable name="LineFlag" select="key('KeyPointH', @pntRef)"/>
			
			<xsl:if test="name()='Start' or name()='End'">
				<xsl:variable name="PointsIL" select="js:PointsDirectPush(string(@pntRef), string(../@oID), string($LineFlag/@lineworkFlag), string(name()), string(name($LineFeat)))"/>
			</xsl:if>
		</xsl:for-each>
		
		<!-- Curve Start & End Export -->
		<xsl:for-each select="//l:CoordGeom/l:Curve/*">
			<xsl:variable name="LineFeat" select="key('KeyLinesH', ../@oID)"/>
			<xsl:variable name="LineFlag" select="key('KeyPointH', @pntRef)"/>
			<xsl:if test="name()='Start' or name()='End'">
				<xsl:variable name="PointsCv" select="js:PointsDirectPush(string(@pntRef), string(../@oID), string($LineFlag/@lineworkFlag), string(name()), string(name($LineFeat)))"/>
			</xsl:if>
		</xsl:for-each>
		
		<!-- Curve Mid Point Export -->
		<xsl:for-each select="//h:CoordGeom/h:Curve/*">
			<xsl:variable name="LineFeat" select="key('KeyLinesH', ../@oID)"/>
			<xsl:variable name="LineFlag" select="key('KeyPointH', @pntRef)"/>
			<xsl:if test="name()='Mid'">
				<xsl:variable name="PointsCvM" select="js:PointsDirectPush(string(@pntRef), string(../@oID), string('CONT'), string(name()), string('Curve'))"/>
			</xsl:if>
		</xsl:for-each>
		
		<!-- Chain Export -->
		<xsl:for-each select="//h:CoordGeom/*/h:Chain">
			<xsl:variable name="LineFeat" select="key('KeyLinesH', ../@oID)"/>
			<xsl:variable name="LineFlag" select="key('KeyPointH', @pntRef)"/>
			<xsl:choose>
				<xsl:when test="boolean(.)">
					<xsl:call-template name="tokenizeString">
						<xsl:with-param name="list" select="."/>
						<xsl:with-param name="delimiter" select="' '"/>
						<xsl:with-param name="lineID" select="../@oID"/>
						<xsl:with-param name="lineType" select="name(..)"/>
					</xsl:call-template>
				</xsl:when>
				<xsl:otherwise/>
			</xsl:choose>
		</xsl:for-each>
		
		<!-- Line Start & End Export -->
		<xsl:for-each select="//l:CoordGeom/l:Line/*">
			<xsl:variable name="LineFeat" select="key('KeyLinesH', ../@oID)"/>
			<xsl:variable name="LineFlag" select="key('KeyPointH', @pntRef)"/>
			<xsl:if test="name()='Start' or name()='End'">
				<xsl:variable name="PointsLn" select="js:PointsFilteredPush(string(@pntRef), string(../@oID), string($LineFlag/@lineworkFlag), string(name()), string('Line'))"/>
			</xsl:if>
		</xsl:for-each>
		
	</xsl:template>

<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
<!--	Tokenize Template																				-->
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
	<xsl:template name="tokenizeString">
		<!--passed template parameter -->
		<xsl:param name="list"/>
		<xsl:param name="delimiter"/>
		<xsl:param name="lineID"/>
		<xsl:param name="lineType"/>
        
		<xsl:choose>
			<xsl:when test="contains($list, $delimiter)">               
				<!-- get everything in front of the first delimiter and send to the JS -->
					<xsl:variable name="PointsCh" select="js:PointsFilteredPush(string(substring-before($list,$delimiter)), string($lineID), string('CONT'), string('Chain'), string($lineType))"/>                    
				<xsl:call-template name="tokenizeString">
					<!-- store anything left in another variable -->
					<xsl:with-param name="list" select="substring-after($list,$delimiter)"/>
					<xsl:with-param name="delimiter" select="$delimiter"/>
					<xsl:with-param name="lineID" select="$lineID"/>
					<xsl:with-param name="lineType" select="$lineType"/>
				</xsl:call-template>
			</xsl:when>
			
			<xsl:otherwise>
				<xsl:choose>
					<xsl:when test="$list = ''">
						<xsl:text/>
					</xsl:when>
					<xsl:otherwise>
						<xsl:variable name="PointsCh" select="js:PointsFilteredPush(string($list), string($lineID), string('CONT'), string('Chain'), string($lineType))"/> 
					</xsl:otherwise>
				</xsl:choose>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
<!--	Java Scripts																					-->
<!--  ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~   -->
	<msxsl:script language="JavaScript" implements-prefix="js">
		<![CDATA[
		
		//-------------------------------------------------------------------------------------------
		// Linework Scripts
		//-------------------------------------------------------------------------------------------
			
			//Array Variables
				var MasterPointList = new Array();
				var MasterLineIdsList = new Array();
				var MasterLineWorkList = new Array();
				var MasterStrtEndList = new Array();
				var MasterLineTypeList = new Array();
			
			// Direct push of information to the Array
				function PointsDirectPush(Point,LineId,LineWork,StrtEnd,LineType)
					{
						MasterPointList.push('Pnt_' + Point + '_Pnt');
						MasterLineIdsList.push(LineId);
						MasterLineWorkList.push(LineWork);
						MasterStrtEndList.push(StrtEnd);
						MasterLineTypeList.push(LineType);
						
						return ("").toString();
					}
			
			// Check for an existing point and only write if there is none			
				function PointsFilteredPush(Point,LineId,LineWork,StrtEnd,LineType)
				{
					var StartVersion = ('Pnt_' + Point + '_Pnt')
					var StartExists = (MasterPointList.toString()).indexOf(StartVersion,0)
					
					if (StartExists == -1)
						{
							MasterPointList.push('Pnt_' + Point + '_Pnt');
							MasterLineIdsList.push(LineId);
							MasterLineWorkList.push(LineWork);
							MasterStrtEndList.push(StrtEnd);
							MasterLineTypeList.push(LineType);
						}
					else	{}
					return ("").toString();
				}			
				
			// Find the current point in the Master Point Array    
				function getIndex(pointId) 
				{	if( MasterPointList.length > 0 ) 
					{	for(var i = 0; i < MasterPointList.length; i++) 
						{	if( 'Pnt_' + pointId + '_Pnt' == MasterPointList[i] ) 
							{	return i;	}	}	} 
					return Number(-1);} 
				
			// Get Current LineID
				function getLineId(arrayIndex) 
				{	var currentLineId = MasterLineIdsList[arrayIndex];
					return (currentLineId).toString();	}
				
			// Get Current LineWork
				function getLineWork(arrayIndex) 
				{	var currentLineWork = MasterLineWorkList[arrayIndex];
					return (currentLineWork).toString();	} 
				
			// Get Current LineStrtEnd
				function getLineStrtEnd(arrayIndex) 
				{	var currentLineStrtEnd = MasterStrtEndList[arrayIndex];
					return (currentLineStrtEnd).toString();	}
				
			// Get Current LineType
				function getLineType(arrayIndex) 
				{	var currentLineType = MasterLineTypeList[arrayIndex];
					return (currentLineType).toString();	}
			
			// LineWork to Operational Code Lookup
				function LineworkLookup(lineType,lineSE)
				{
					// Linework Lookup Table
						var lineOpCode = (lineType + lineSE)
					
					var lineworkCCad = {
					"LineStart"         : function() {return "" 	;},
					"LineEnd"           : function() {return "" 	;},
					"SplineStart"       : function() {return ('61' + '\r\n');},
					"SplineChain"       : function() {return "" 	;},
					"SplineEnd"	        : function() {return ('62' + '\r\n');},
					"BestFitCurveStart" : function() {return ('61' + '\r\n');},
					"BestFitCurveChain" : function() {return "" 	;},
					"BestFitCurveEnd"   : function() {return ('62' + '\r\n');},
					"CurveStart"        : function() {return ('60' + '\r\n');},
					"CurveMid"          : function() {return "" 	;},
					"CurveEnd"          : function() {return "" 	;}
					};
					
				var returnlineflag = lineworkCCad[lineOpCode]();
				return (returnlineflag).toString();
				}	
			
			
			
		//-------------------------------------------------------------------------------------------
		// Conversion Scripts
		//-------------------------------------------------------------------------------------------
			
			// Convert TPS DMS to DecDeg
				function ConvertDMStoDD(TPS)
				{
					var DMS = Math.abs(TPS);
					var wholeDeg = Math.floor(DMS);
					var remainderDeg = DMS - wholeDeg;
					var fullMins = remainderDeg * 100;
					var wholeMins = Math.floor(fullMins);
					var remainderMins = fullMins - wholeMins;
					var fullSec = remainderMins * 100;
					var dec_mins = wholeMins / 60;
					var dec_sec = fullSec / 60 / 60;
					var decDeg = wholeDeg + dec_mins + dec_sec;	
					
					return (decDeg.toFixed(20)).toString();
				}
			
			// Convert XML Time to Aus Date
				function XMLTimeAuDate(xmlDate)
				{
					var AuYear = xmlDate.substring(2,4);
					var AuMonth = xmlDate.substring(5,7);
					var AuDay = xmlDate.substring(8,10);
					var fwdslash = "\/";
					var AuYearM = AuYear *1;
					var AuYear2000 = AuYearM + 2000;
					return (AuDay + fwdslash + AuMonth + fwdslash + AuYear2000).toString();
				}
			
			// Convert XML Time to Aus Time
				function XMLTimeAuTime(xmlTime)
				{
					var AuTime = xmlTime.substring(11,22);
					return (AuTime).toString();
				}
			
			// Convert Dec Deg to DMS Text
					
				function DDtoDMSTxt(decDeg,DecPl)
				{
					var Degree = "\u00b0";
					var Minute = "\u0027";
					var Second = "\u0022";
					var DD = Math.abs(decDeg);
					var wholeDeg = Math.floor(DD);
					var remainderDeg = DD - wholeDeg;
					var fullMins = remainderDeg * 60;
					var wholeMins = Math.floor(fullMins);
					var remainderMins = fullMins - wholeMins;
					var fullSec = remainderMins * 60;
					
					if (decDeg < 0)
						{ var rtnDeg = (wholeDeg * -1) + Degree;	}
					else
						{ var rtnDeg = wholeDeg + Degree;	}		
					
					if ( wholeMins < 10 )
						{var rtnMin = ' 0' + wholeMins + Minute;	}
					else
						{var rtnMin = ' ' + wholeMins + Minute;	}
					
					if ( fullSec < 10 )
						{var rtnSec = ' 0' + (fullSec.toFixed(DecPl)) + Second;	}
					else
						{var rtnSec = ' ' + (fullSec.toFixed(DecPl)) + Second;	}
					
					return (rtnDeg + rtnMin + rtnSec).toString();
				}
		
		//-------------------------------------------------------------------------------------------
		// Check if Current Target Height is the same as the previous and decide if to print
		//-------------------------------------------------------------------------------------------
			
			// Define Global currentTH Variables
			
			var currentTH = ""
			var ModTH = ""
			
			// Set Current Target Ht to Default Value after Station Setup
			
			function ResetCurrentTH(resetHt)
			{
				currentTH = resetHt;
				return ('').toString();   
			}
			
			// Check Current Target Ht to print
			
			function CurrentTHCheck(MeasTH)
			{
				if (MeasTH != currentTH)
					{ModTH = (' 5' + '\t' + '\t' + MeasTH.toFixed(3) + '\r'), currentTH = MeasTH;}
				else
					{ModTH = ""}
				
				return (ModTH).toString();
			}
		
		
		
		//-------------------------------------------------------------------------------------------
		// Check if Current Antenna Height is the same as the previous and decide if to print
		//-------------------------------------------------------------------------------------------
			
			// Define Global currentAnt Variables
			
			var currentAnt = ""
			var ModAnt = ""
			
			// Set Current Antenna Ht to Default Value after Station Setup
			
			function ResetCurrentAnt(resetAntHt)
			{
				currentAnt = resetAntHt;
				return ('').toString();   
			}
			
			// Check Current Ant Ht to print
			
			function CurrentAntCheck(MeasAnt)
			{
				if (MeasAnt != currentAnt)                                                            
					{ModAnt = ('\r\n' + '//' + '\t' + '\t' + 'Antenna Height Change ' + MeasAnt.toFixed(3) + '\r\n' + '\r\n'), currentAnt = MeasAnt;} 
				else
					{ModAnt = ""}                                                                    	
				return (ModAnt).toString();
			}
		
		
		
		//-------------------------------------------------------------------------------------------
		// Check if the TPS Correction UniqueID is the same as the previous and decide if to print
		//-------------------------------------------------------------------------------------------
			
			// Define Global Current TPS Correction Status Variables
			
			var TPSCorrectionID = "";
			var TPSCorrStatus = "";
			
			
			// Set Current TPS Correction Status to Default Value after Station Setup
			
			function ResetTPSCorrStatus()
			{
				TPSCorrectionID = '0';
				return ('').toString();
			}
			
			// Check Current TPS Correction UniqueID to print
			
			function TPSCorr(uniqueID)
			{
				if (TPSCorrectionID != uniqueID)
					{TPSCorrStatus = '1', TPSCorrectionID = uniqueID;}
				else
					{TPSCorrStatus = '0'}
				
				return (TPSCorrStatus).toString();
			}		
		
		
		//-------------------------------------------------------------------------------------------
		// Check if EDM & ATR Settings are changed and decide if to print
		//-------------------------------------------------------------------------------------------
			
			// Define Global checkEDMATR Variables and set initial value to blank
			
			var currentedmKind = ""
			var currentedmMode = ""
			var currentatrMode = ""
			var currentreflectorName = ""
			var currentreflectorConstant = ""
			var changededmKind = ""
			var changededmMode = ""
			var changedatrMode = ""
			var changedreflectorName = ""
			var changedreflectorConstant = ""
			var currentInfo = ""
					
			// Set Current EDM & ATR to Default Values after Station Setup
			
			function ResetEDMATR()
			{
				currentedmKind = ""
				currentedmMode = ""
				currentatrMode = ""
				currentreflectorName = ""
				currentreflectorConstant = ""
				changededmKind = ""
				changededmMode = ""
				changedatrMode = ""
				changedreflectorName = ""
				changedreflectorConstant = ""
				return ('').toString();  
			}
			
			// Check Current EDM & ATR
			
			function checkEDMATR(edmKind,edmMode,atrMode,refName,refCon)
			{
				if (edmKind == currentedmKind && edmMode == currentedmMode && atrMode == currentatrMode && refName == currentreflectorName && refCon == currentreflectorConstant)
					{currentInfo = "";}	
					
				else 
					{
					// Check for change and write new values
						if (edmKind == currentedmKind) {changededmKind = '';} else {changededmKind = ( '  EDM: ' + edmKind );}
						if (edmMode == currentedmMode) {changededmMode = '';} else {changededmMode = ( '  Mode: ' + edmMode );}
						if (atrMode == currentatrMode) {changedatrMode = '';} else {changedatrMode = ( '  ATR: ' + atrMode );}
						if (refName == currentreflectorName) {changedreflectorName = '';} else {changedreflectorName = ( '  Reflector: ' + refName );}
						if (refCon == currentreflectorConstant) {changedreflectorConstant = '';} else {changedreflectorConstant = ( '  Constant: ' + ( refCon * 1000 ).toFixed(1) + 'mm' );}
						
					// Write Change line and send updated values to the global variable
						currentInfo = ( '//' + '\t' + '\t' + 'TPS Config Change:' + changededmKind + changededmMode + changedatrMode + changedreflectorName + changedreflectorConstant + '\r\n' ),
						currentedmKind = edmKind, 
						currentedmMode = edmMode, 
						currentatrMode = atrMode, 
						currentreflectorName = refName, 
						currentreflectorConstant = refCon;
					}	
				return (currentInfo).toString();
			}
		
		//-------------------------------------------------------------------------------------------
		// Justify Text
		//-------------------------------------------------------------------------------------------
		
		function jText(origText,TxtLen,jLR,Trunc)
		{
			if (origText.length > TxtLen)
				{ 
					if (Trunc == 'N')
						{	var jText = origText;	}
					else
						{	if (Trunc == 'L')
								{	var jText = origText.substring(0,TxtLen);	}
							else
								{	var jText = origText.substring(origText.length - TxtLen,origText.length);	}
						}
				}
			else
				{
					var fillchar = "                                              ";			//Set the fill character
					var jstr = fillchar.substring(0,TxtLen - origText.length);		//Create a string of fill characters at the fill length
					if (jLR == 'L')
						{ var jText = origText + jstr; }
					else
						{ var jText = jstr + origText; }
				}
			return (jText).toString();
		}
		
		//-------------------------------------------------------------------------------------------
		// Justify Distances
		//-------------------------------------------------------------------------------------------
		
		function jDist(origDist,TxtLen,jLR,DecPl)
		{
			var fillchar = "                    ";
			var decPlaces = Math.pow(10,DecPl);
			var origDistRound = Math.round(origDist * decPlaces);
			var rDist = (origDistRound / decPlaces).toFixed(DecPl)
			var jstr = fillchar.substring(0,TxtLen - rDist.toString().length);
			if (jLR == 'L')
				{ var jrDist = rDist + jstr; }
			else
				{ var jrDist = jstr + rDist; }
			return (jrDist).toString();
		}
		
		
		//-------------------------------------------------------------------------------------------
		// Justify Angles
		//-------------------------------------------------------------------------------------------
			
		function jAngle(origAngle,TxtLen,jLR,DecPl)
		{
			var fillchar = "                                        ";
			var SecDecPl = DecPl - 4;
			var DMS = Math.abs(origAngle);
			var wholeDeg = Math.floor(DMS);
			var remainderDeg = DMS - wholeDeg;
			var fullMins = remainderDeg * 100;
			var wholeMins = Math.floor(fullMins);
			var remainderMins = fullMins - wholeMins;
			var fullSec = remainderMins * 100;
			var wholeSec = Math.floor(fullSec);
			var remainderSec = fullSec - wholeSec;
			var decPlaces = Math.pow(10,SecDecPl);
			var fulldecPlac = remainderSec * decPlaces;
			var roundDecPl = Math.round(fulldecPlac);
			var RsemifinalSec = ''
			var RfinalSec = ''
			var RsemifinalMin = ''
			var RfinalMin = ''
			var RsemifinalDeg = ''
			var RfinalDeg = ''
			var rAngle = ''
			var rsAngle = ''
			
			if (roundDecPl == decPlaces)
				{RsemifinalSec = (wholeSec + 1);}
			else
				{RsemifinalSec = wholeSec + (roundDecPl / decPlaces);}
			
			if (RsemifinalSec != 60)
				{RfinalSec = RsemifinalSec, RsemifinalMin = wholeMins;}  	//Check if Sec now 60, Change to 0 and add 1 to the Min
			else
				{RfinalSec = 0, RsemifinalMin = wholeMins + 1;}
			
			if (RsemifinalMin !=60)										//Check if Mins now 60, Change to 0 and add 1 to the Deg
				{RfinalMin = RsemifinalMin, RsemifinalDeg = wholeDeg;}
			else
				{RfinalMin = 0, RsemifinalDeg = wholeDeg + 1;}
			
			if (RsemifinalDeg !=360)										//Check if Deg is now 360 and change to 0
				{RfinalDeg = RsemifinalDeg;}
			else
				{RfinalDeg = 0;}			
			
			var newDeg = RfinalDeg;
			var newMin = RfinalMin / 100;
			var newSec = RfinalSec / 10000;
			if(origAngle < 0)
				{
				rAngle = ((newDeg + newMin + newSec).toFixed(SecDecPl + 4)) * -1;
				}
			else
				{
				rAngle = ((newDeg + newMin + newSec).toFixed(SecDecPl + 4));
				}
			
			var jstr = fillchar.substring(0,TxtLen - rAngle.toString().length);
			if (jLR == 'L')
				{ var jrAngle = rAngle + jstr; }
			else
				{ var jrAngle = jstr + rAngle; }
			
			return (jrAngle).toString();
		}		
		
		//-------------------------------------------------------------------------------------------
		// PU 95%CL Calcs
		//-------------------------------------------------------------------------------------------
			
			function PU95cl(SUPoint,SUBase)
			{
				var SUPoint2 = Math.pow(SUPoint, 2);
				var SUBase2 = Math.pow(SUBase, 2);
				var PUPosit = Math.sqrt(SUPoint2 + SUBase2);
				return (PUPosit).toString();
			}
		
		//-------------------------------------------------------------------------------------------
		// Tinnable Flag
		//-------------------------------------------------------------------------------------------
			
			var TinExportFlag = '';
			
			function TinnableFlag(TinFlag)
			{
				if (TinFlag == 0)
					{ TinExportFlag = 40 + '\r\n'; }
				else
					{
						if (TinFlag == '2')
							{ TinExportFlag = 30 + '\r\n'; }
						else
							{TinExportFlag = '';}
					}
				return ('').toString();
			}
			
			function GetTinnableFlag()
			{
			return (TinExportFlag).toString();
			}
			
		]]>
	</msxsl:script>

</xsl:stylesheet>
